/*
 * Copyright 2017-2019 John A. De Goes and the ZIO Contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package zio.stream

import com.github.ghik.silencer.silent
import zio._

/**
 * A `ZStreamChunk[R, E, A]` represents an effectful stream that can produce values of
 * type `A`, or potentially fail with a value of type `E`.
 *
 * `ZStreamChunk` differs from `ZStream` in that elements in the stream are processed
 * in batches, which is orders of magnitude more efficient than dealing with each
 * element individually.
 *
 * `ZStreamChunk` is particularly suited for situations where you are dealing with values
 * of primitive types, e.g. those coming off a `java.io.InputStream`
 */
class ZStreamChunk[-R, +E, +A](val chunks: ZStream[R, E, Chunk[A]]) { self =>
  import ZStream.Pull

  /**
   * Concatenates with another stream in strict order
   */
  final def ++[R1 <: R, E1 >: E, A1 >: A](that: ZStreamChunk[R1, E1, A1]): ZStreamChunk[R1, E1, A1] =
    ZStreamChunk(chunks ++ that.chunks)

  /**
   * Allows a faster producer to progress independently of a slower consumer by buffering
   * up to `capacity` chunks in a queue.
   *
   * @note Prefer capacities that are powers of 2 for better performance.
   */
  final def buffer(capacity: Int): ZStreamChunk[R, E, A] =
    ZStreamChunk(chunks.buffer(capacity))

  /**
   * Allows a faster producer to progress independently of a slower consumer by buffering
   * elements into a dropping queue.
   *
   * @note Prefer capacities that are powers of 2 for better performance.
   */
  final def bufferDropping(capacity: Int): ZStreamChunk[R, E, A] =
    ZStreamChunk(chunks.bufferDropping(capacity))

  /**
   * Allows a faster producer to progress independently of a slower consumer by buffering
   * elements into a sliding queue.
   *
   * @note Prefer capacities that are powers of 2 for better performance.
   */
  final def bufferSliding(capacity: Int): ZStreamChunk[R, E, A] =
    ZStreamChunk(chunks.bufferSliding(capacity))

  /**
   * Allows a faster producer to progress independently of a slower consumer by buffering
   * elements into an unbounded queue.
   */
  final def bufferUnbounded: ZStreamChunk[R, E, A] =
    ZStreamChunk(chunks.bufferUnbounded)

  /**
   * Collects a filtered, mapped subset of the stream.
   */
  final def collect[B](p: PartialFunction[A, B]): ZStreamChunk[R, E, B] =
    ZStreamChunk(self.chunks.map(chunk => chunk.collect(p)))

  /**
   * Drops the specified number of elements from this stream.
   */
  def drop(n: Int): ZStreamChunk[R, E, A] =
    ZStreamChunk {
      ZStream[R, E, Chunk[A]] {
        for {
          chunks     <- self.chunks.process
          counterRef <- Ref.make(n).toManaged_
          pull = {
            def go: Pull[R, E, Chunk[A]] =
              chunks.flatMap { chunk =>
                counterRef.get.flatMap { cnt =>
                  if (cnt <= 0) Pull.emit(chunk)
                  else {
                    val remaining = chunk.drop(cnt)
                    val dropped   = chunk.length - remaining.length
                    counterRef.set(cnt - dropped) *>
                      (if (remaining.isEmpty) go else Pull.emit(remaining))
                  }
                }
              }

            go
          }
        } yield pull
      }
    }

  /**
   * Drops all elements of the stream for as long as the specified predicate
   * evaluates to `true`.
   */
  def dropWhile(pred: A => Boolean): ZStreamChunk[R, E, A] =
    ZStreamChunk {
      ZStream[R, E, Chunk[A]] {
        for {
          chunks          <- self.chunks.process
          keepDroppingRef <- Ref.make(true).toManaged_
          pull = {
            def go: Pull[R, E, Chunk[A]] =
              chunks.flatMap { chunk =>
                keepDroppingRef.get.flatMap { keepDropping =>
                  if (!keepDropping) Pull.emit(chunk)
                  else {
                    val remaining = chunk.dropWhile(pred)
                    val empty     = remaining.length <= 0

                    if (empty) go
                    else keepDroppingRef.set(false).as(remaining)
                  }
                }
              }

            go
          }
        } yield pull
      }
    }

  /**
   * Filters this stream by the specified predicate, retaining all elements for
   * which the predicate evaluates to true.
   */
  def filter(pred: A => Boolean): ZStreamChunk[R, E, A] =
    ZStreamChunk[R, E, A](self.chunks.map(_.filter(pred)))

  /**
   * Filters this stream by the specified predicate, removing all elements for
   * which the predicate evaluates to true.
   */
  final def filterNot(pred: A => Boolean): ZStreamChunk[R, E, A] = filter(!pred(_))

  /**
   * Returns a stream made of the concatenation in strict order of all the streams
   * produced by passing each element of this stream to `f0`
   */
  final def flatMap[R1 <: R, E1 >: E, B](f0: A => ZStreamChunk[R1, E1, B]): ZStreamChunk[R1, E1, B] =
    ZStreamChunk(
      chunks.flatMap(_.map(f0).fold[ZStream[R1, E1, Chunk[B]]](ZStream.empty)((acc, el) => acc ++ el.chunks))
    )

  /**
   * Returns a stream made of the concatenation of all the chunks in this stream
   */
  def flattenChunks: ZStream[R, E, A] = chunks.flatMap(ZStream.fromChunk)

  /**
   * Executes a pure fold over the stream of values - reduces all elements in the stream to a value of type `S`.
   * See [[ZStream.fold]]
   */
  def fold[A1 >: A, S](s: S)(f: (S, A1) => S): ZIO[R, E, S] =
    chunks
      .foldWhileManagedM[R, E, Chunk[A1], S](s)(_ => true) { (s: S, as: Chunk[A1]) =>
        as.foldM[Any, Nothing, S](s)((s, a) => ZIO.succeed(f(s, a)))
      }
      .use(ZIO.succeed)

  /**
   * Executes an effectful fold over the stream of values.
   * See [[ZStream.foldM]]
   */
  final def foldM[R1 <: R, E1 >: E, A1 >: A, S](s: S)(f: (S, A1) => ZIO[R1, E1, S]): ZIO[R1, E1, S] =
    chunks
      .foldWhileManagedM[R1, E1, Chunk[A1], S](s)(_ => true) { (s, as) =>
        as.foldM(s)(f)
      }
      .use(ZIO.succeed)

  /**
   * Executes a pure fold over the stream of values.
   * Returns a Managed value that represents the scope of the stream.
   * See [[ZStream.foldManaged]]
   */
  final def foldManaged[A1 >: A, S](s: S)(f: (S, A1) => S): ZManaged[R, E, S] =
    chunks
      .foldWhileManagedM[R, E, Chunk[A1], S](s)(_ => true) { (s: S, as: Chunk[A1]) =>
        as.foldM[Any, Nothing, S](s)((s, a) => ZIO.succeed(f(s, a)))
      }

  /**
   * Executes an effectful fold over the stream of values.
   * Returns a Managed value that represents the scope of the stream.
   * See [[ZStream.foldManagedM]]
   */
  final def foldManagedM[R1 <: R, E1 >: E, A1 >: A, S](s: S)(f: (S, A1) => ZIO[R1, E1, S]): ZManaged[R1, E1, S] =
    chunks.foldWhileManagedM[R1, E1, Chunk[A1], S](s)(_ => true) { (s, as) =>
      as.foldM(s)(f)
    }

  /**
   * Reduces the elements in the stream to a value of type `S`.
   * Stops the fold early when the condition is not fulfilled.
   * See [[ZStream.foldWhile]]
   */
  final def foldWhile[A1 >: A, S](s: S)(cont: S => Boolean)(f: (S, A1) => S): ZIO[R, E, S] =
    chunks
      .foldWhileManagedM[R, E, Chunk[A1], S](s)(cont) { (s: S, as: Chunk[A1]) =>
        as.foldWhileM(s)(cont)((s, a) => ZIO.succeed(f(s, a)))
      }
      .use(ZIO.succeed)

  /**
   * Executes an effectful fold over the stream of values.
   * Stops the fold early when the condition is not fulfilled.
   * See [[ZStream.foldWhileM]]
   */
  final def foldWhileM[R1 <: R, E1 >: E, A1 >: A, S](
    s: S
  )(cont: S => Boolean)(f: (S, A1) => ZIO[R1, E1, S]): ZIO[R1, E1, S] =
    chunks
      .foldWhileManagedM[R1, E1, Chunk[A1], S](s)(cont) { (s, as) =>
        as.foldWhileM(s)(cont)(f)
      }
      .use(ZIO.succeed)

  /**
   * Executes a pure fold over the stream of values.
   * Stops the fold early when the condition is not fulfilled.
   * See [[ZStream.foldWhileManaged]]
   */
  def foldWhileManaged[A1 >: A, S](s: S)(cont: S => Boolean)(f: (S, A1) => S): ZManaged[R, E, S] =
    chunks.foldWhileManagedM[R, E, Chunk[A1], S](s)(cont) { (s: S, as: Chunk[A1]) =>
      as.foldWhileM[Any, Nothing, S](s)(cont)((s, a) => ZIO.succeed(f(s, a)))
    }

  /**
   * Executes an effectful fold over the stream of values.
   * Stops the fold early when the condition is not fulfilled.
   * See [[ZStream.foldWhileManagedM]]
   */
  final def foldWhileManagedM[R1 <: R, E1 >: E, A1 >: A, S](
    s: S
  )(cont: S => Boolean)(f: (S, A1) => ZIO[R1, E1, S]): ZManaged[R1, E1, S] =
    chunks.foldWhileManagedM[R1, E1, Chunk[A1], S](s)(cont) { (s, as) =>
      as.foldWhileM(s)(cont)(f)
    }

  /**
   * Consumes all elements of the stream, passing them to the specified callback.
   */
  final def foreach[R1 <: R, E1 >: E](f: A => ZIO[R1, E1, Unit]): ZIO[R1, E1, Unit] =
    foreachWhile[R1, E1](f(_).as(true))

  /**
   * Consumes elements of the stream, passing them to the specified callback,
   * and terminating consumption when the callback returns `false`.
   */
  final def foreachWhile[R1 <: R, E1 >: E](f: A => ZIO[R1, E1, Boolean]): ZIO[R1, E1, Unit] =
    chunks.foreachWhile[R1, E1] { as =>
      as.foldWhileM(true)(identity) { (p, a) =>
        if (p) f(a)
        else IO.succeed(p)
      }
    }

  /**
   * Returns a stream made of the elements of this stream transformed with `f0`
   */
  def map[B](f: A => B): ZStreamChunk[R, E, B] =
    ZStreamChunk(chunks.map(_.map(f)))

  /**
   * Statefully maps over the elements of this stream to produce new elements.
   */
  final def mapAccum[S1, B](s1: S1)(f1: (S1, A) => (S1, B)): ZStreamChunk[R, E, B] =
    ZStreamChunk(chunks.mapAccum(s1)((s1: S1, as: Chunk[A]) => as.mapAccum(s1)(f1)))

  /**
   * Maps each element to a chunk and flattens the chunks into the output of
   * this stream.
   */
  def mapConcatChunk[B](f: A => Chunk[B]): ZStreamChunk[R, E, B] =
    ZStreamChunk(chunks.map(_.flatMap(f)))

  /**
   * Maps each element to an iterable and flattens the iterable into the output of
   * this stream.
   */
  def mapConcat[B](f: A => Iterable[B]): ZStreamChunk[R, E, B] =
    mapConcatChunk(f andThen Chunk.fromIterable)

  /**
   * Maps over elements of the stream with the specified effectful function.
   */
  final def mapM[R1 <: R, E1 >: E, B](f0: A => ZIO[R1, E1, B]): ZStreamChunk[R1, E1, B] =
    ZStreamChunk(chunks.mapM(_.mapM(f0)))

  final def process =
    for {
      chunks   <- self.chunks.process
      chunkRef <- Ref.make[Chunk[A]](Chunk.empty).toManaged_
      indexRef <- Ref.make(0).toManaged_
      pull = {
        def go: Pull[R, E, A] =
          chunkRef.get.flatMap { chunk =>
            indexRef.get.flatMap { index =>
              if (index < chunk.length) indexRef.set(index + 1).as(chunk(index))
              else
                chunks.flatMap { chunk =>
                  chunkRef.set(chunk) *> indexRef.set(0) *> go
                }
            }
          }

        go
      }
    } yield pull

  /**
   * Runs the sink on the stream to produce either the sink's result or an error.
   */
  final def run[R1 <: R, E1 >: E, A0, A1 >: A, B](sink: ZSink[R1, E1, A0, Chunk[A1], B]): ZIO[R1, E1, B] =
    chunks.run(sink)

  /**
   * Takes the specified number of elements from this stream.
   */
  def take(n: Int): ZStreamChunk[R, E, A] =
    ZStreamChunk {
      ZStream[R, E, Chunk[A]] {
        for {
          chunks     <- self.chunks.process
          counterRef <- Ref.make(n).toManaged_
          pull = counterRef.get.flatMap { cnt =>
            if (cnt <= 0) Pull.end
            else
              for {
                chunk <- chunks
                taken = chunk.take(cnt)
                _     <- counterRef.set(cnt - taken.length)
              } yield taken
          }
        } yield pull
      }
    }

  /**
   * Takes all elements of the stream for as long as the specified predicate
   * evaluates to `true`.
   */
  def takeWhile(pred: A => Boolean): ZStreamChunk[R, E, A] =
    ZStreamChunk {
      ZStream[R, E, Chunk[A]] {
        for {
          chunks  <- self.chunks.process
          doneRef <- Ref.make(false).toManaged_
          pull = doneRef.get.flatMap { done =>
            if (done) Pull.end
            else
              for {
                chunk     <- chunks
                remaining = chunk.takeWhile(pred)
                _         <- doneRef.set(true).when(remaining.length < chunk.length)
              } yield remaining
          }
        } yield pull
      }
    }

  /**
   * Adds an effect to consumption of every element of the stream.
   */
  final def tap[R1 <: R, E1 >: E](f0: A => ZIO[R1, E1, _]): ZStreamChunk[R1, E1, A] =
    ZStreamChunk(chunks.tap[R1, E1] { as =>
      as.mapM_(f0)
    })

  @silent("never used")
  def toInputStream(implicit ev0: E <:< Throwable, ev1: A <:< Byte): ZManaged[R, E, java.io.InputStream] =
    for {
      runtime <- ZIO.runtime[R].toManaged_
      pull    <- process
      javaStream = new java.io.InputStream {
        override def read(): Int = {
          val exit = runtime.unsafeRunSync[Option[Throwable], Byte](pull.asInstanceOf[Pull[R, Throwable, Byte]])
          ZStream.exitToInputStreamRead(exit)
        }
      }
    } yield javaStream

  /**
   * Converts the stream to a managed queue. After managed the queue is used,
   * the queue will never again produce chunks and should be discarded.
   */
  final def toQueue[E1 >: E, A1 >: A](capacity: Int = 2): ZManaged[R, Nothing, Queue[Take[E1, Chunk[A1]]]] =
    chunks.toQueue(capacity)

  /**
   * Converts the stream into an unbounded managed queue. After the managed queue
   * is used, the queue will never again produce values and should be discarded.
   */
  final def toQueueUnbounded[E1 >: E, A1 >: A]: ZManaged[R, Nothing, Queue[Take[E1, Chunk[A1]]]] =
    chunks.toQueueUnbounded

  /**
   * Converts the stream to a managed queue and immediately consume its
   * elements.
   */
  final def toQueueWith[R1 <: R, E1 >: E, A1 >: A, Z](
    f: Queue[Take[E1, Chunk[A1]]] => ZIO[R1, E1, Z],
    capacity: Int = 1
  ): ZIO[R1, E1, Z] =
    toQueue[E1, A1](capacity).use(f)

  /**
   * Zips this stream together with the index of elements of the stream across chunks.
   */
  final def zipWithIndex: ZStreamChunk[R, E, (A, Int)] =
    self.mapAccum(0)((index, a) => (index + 1, (a, index)))
}

object ZStreamChunk {

  /**
   * The default chunk size used by the various combinators and constructors of [[ZStreamChunk]].
   */
  final val DefaultChunkSize: Int = 4096

  /**
   * The empty stream of chunks
   */
  final val empty: StreamChunk[Nothing, Nothing] =
    new StreamEffectChunk(StreamEffect.empty)

  /**
   * Creates a `ZStreamChunk` from a stream of chunks
   */
  final def apply[R, E, A](chunkStream: ZStream[R, E, Chunk[A]]): ZStreamChunk[R, E, A] =
    new ZStreamChunk[R, E, A](chunkStream)

  /**
   * Creates a `ZStreamChunk` from a variable list of chunks
   */
  final def fromChunks[A](as: Chunk[A]*): StreamChunk[Nothing, A] =
    new StreamEffectChunk(StreamEffect.fromIterable(as))

  /**
   * Creates a `ZStreamChunk` from a chunk
   */
  final def succeed[A](as: Chunk[A]): StreamChunk[Nothing, A] =
    new StreamEffectChunk(StreamEffect.succeed(as))
}
