package zio.aws.kinesis.model
import zio.ZIO
import zio.aws.core.{AwsError, BuilderHelper}
import java.time.Instant
import zio.aws.kinesis.model.primitives.{
  PartitionKey,
  Data,
  Timestamp,
  SequenceNumber
}
import scala.jdk.CollectionConverters.*
final case class Record(
    sequenceNumber: SequenceNumber,
    approximateArrivalTimestamp: Option[Timestamp] = None,
    data: Data,
    partitionKey: PartitionKey,
    encryptionType: Option[zio.aws.kinesis.model.EncryptionType] = None
) {
  def buildAwsValue(): software.amazon.awssdk.services.kinesis.model.Record = {
    import Record.zioAwsBuilderHelper.BuilderOps
    software.amazon.awssdk.services.kinesis.model.Record
      .builder()
      .sequenceNumber(SequenceNumber.unwrap(sequenceNumber): java.lang.String)
      .optionallyWith(
        approximateArrivalTimestamp.map(value =>
          Timestamp.unwrap(value): Instant
        )
      )(_.approximateArrivalTimestamp)
      .data(
        software.amazon.awssdk.core.SdkBytes
          .fromByteArrayUnsafe(data.toArray[Byte])
      )
      .partitionKey(PartitionKey.unwrap(partitionKey): java.lang.String)
      .optionallyWith(encryptionType.map(value => value.unwrap))(
        _.encryptionType
      )
      .build()
  }
  def asReadOnly: zio.aws.kinesis.model.Record.ReadOnly =
    zio.aws.kinesis.model.Record.wrap(buildAwsValue())
}
object Record {
  private lazy val zioAwsBuilderHelper
      : BuilderHelper[software.amazon.awssdk.services.kinesis.model.Record] =
    BuilderHelper.apply
  trait ReadOnly {
    def asEditable: zio.aws.kinesis.model.Record = zio.aws.kinesis.model.Record(
      sequenceNumber,
      approximateArrivalTimestamp.map(value => value),
      data,
      partitionKey,
      encryptionType.map(value => value)
    )
    def sequenceNumber: SequenceNumber
    def approximateArrivalTimestamp: Option[Timestamp]
    def data: Data
    def partitionKey: PartitionKey
    def encryptionType: Option[zio.aws.kinesis.model.EncryptionType]
    def getSequenceNumber: ZIO[Any, Nothing, SequenceNumber] =
      ZIO.succeed(sequenceNumber)
    def getApproximateArrivalTimestamp: ZIO[Any, AwsError, Timestamp] =
      AwsError.unwrapOptionField(
        "approximateArrivalTimestamp",
        approximateArrivalTimestamp
      )
    def getData: ZIO[Any, Nothing, Data] = ZIO.succeed(data)
    def getPartitionKey: ZIO[Any, Nothing, PartitionKey] =
      ZIO.succeed(partitionKey)
    def getEncryptionType
        : ZIO[Any, AwsError, zio.aws.kinesis.model.EncryptionType] =
      AwsError.unwrapOptionField("encryptionType", encryptionType)
  }
  private final class Wrapper(
      impl: software.amazon.awssdk.services.kinesis.model.Record
  ) extends zio.aws.kinesis.model.Record.ReadOnly {
    override val sequenceNumber: SequenceNumber =
      zio.aws.kinesis.model.primitives.SequenceNumber(impl.sequenceNumber())
    override val approximateArrivalTimestamp: Option[Timestamp] = scala
      .Option(impl.approximateArrivalTimestamp())
      .map(value => zio.aws.kinesis.model.primitives.Timestamp(value))
    override val data: Data = zio.aws.kinesis.model.primitives
      .Data(zio.Chunk.fromArray(impl.data().asByteArrayUnsafe()))
    override val partitionKey: PartitionKey =
      zio.aws.kinesis.model.primitives.PartitionKey(impl.partitionKey())
    override val encryptionType: Option[zio.aws.kinesis.model.EncryptionType] =
      scala
        .Option(impl.encryptionType())
        .map(value => zio.aws.kinesis.model.EncryptionType.wrap(value))
  }
  def wrap(
      impl: software.amazon.awssdk.services.kinesis.model.Record
  ): zio.aws.kinesis.model.Record.ReadOnly = new Wrapper(impl)
}
