package zio.aws.cloudfront.model
import java.time.Instant
import zio.prelude.data.Optional
import zio.aws.cloudfront.model.primitives.Timestamp
import zio.aws.core.{AwsError, BuilderHelper}
import zio.ZIO
import scala.jdk.CollectionConverters._
final case class KeyValueStore(
    name: String,
    id: String,
    comment: String,
    arn: String,
    status: Optional[String] = Optional.Absent,
    lastModifiedTime: Timestamp
) {
  def buildAwsValue()
      : software.amazon.awssdk.services.cloudfront.model.KeyValueStore = {
    import KeyValueStore.zioAwsBuilderHelper.BuilderOps
    software.amazon.awssdk.services.cloudfront.model.KeyValueStore
      .builder()
      .name(name: java.lang.String)
      .id(id: java.lang.String)
      .comment(comment: java.lang.String)
      .arn(arn: java.lang.String)
      .optionallyWith(status.map(value => value: java.lang.String))(_.status)
      .lastModifiedTime(Timestamp.unwrap(lastModifiedTime): Instant)
      .build()
  }
  def asReadOnly: zio.aws.cloudfront.model.KeyValueStore.ReadOnly =
    zio.aws.cloudfront.model.KeyValueStore.wrap(buildAwsValue())
}
object KeyValueStore {
  private lazy val zioAwsBuilderHelper: BuilderHelper[
    software.amazon.awssdk.services.cloudfront.model.KeyValueStore
  ] = BuilderHelper.apply
  trait ReadOnly {
    def asEditable: zio.aws.cloudfront.model.KeyValueStore =
      zio.aws.cloudfront.model.KeyValueStore(
        name,
        id,
        comment,
        arn,
        status.map(value => value),
        lastModifiedTime
      )
    def name: String
    def id: String
    def comment: String
    def arn: String
    def status: Optional[String]
    def lastModifiedTime: Timestamp
    def getName: ZIO[Any, Nothing, String] = ZIO.succeed(name)
    def getId: ZIO[Any, Nothing, String] = ZIO.succeed(id)
    def getComment: ZIO[Any, Nothing, String] = ZIO.succeed(comment)
    def getArn: ZIO[Any, Nothing, String] = ZIO.succeed(arn)
    def getStatus: ZIO[Any, AwsError, String] =
      AwsError.unwrapOptionField("status", status)
    def getLastModifiedTime: ZIO[Any, Nothing, Timestamp] =
      ZIO.succeed(lastModifiedTime)
  }
  private final class Wrapper(
      impl: software.amazon.awssdk.services.cloudfront.model.KeyValueStore
  ) extends zio.aws.cloudfront.model.KeyValueStore.ReadOnly {
    override val name: String = impl.name(): String
    override val id: String = impl.id(): String
    override val comment: String = impl.comment(): String
    override val arn: String = impl.arn(): String
    override val status: Optional[String] = zio.aws.core.internal
      .optionalFromNullable(impl.status())
      .map(value => value: String)
    override val lastModifiedTime: Timestamp =
      zio.aws.cloudfront.model.primitives.Timestamp(impl.lastModifiedTime())
  }
  def wrap(
      impl: software.amazon.awssdk.services.cloudfront.model.KeyValueStore
  ): zio.aws.cloudfront.model.KeyValueStore.ReadOnly = new Wrapper(impl)
}
