package zio.aws.athena.model
import zio.aws.athena.model.primitives.{
  ParametersMapValue,
  KeyString,
  DescriptionString,
  NameString
}
import zio.ZIO
import zio.aws.core.{AwsError, BuilderHelper}
import scala.jdk.CollectionConverters.*
final case class Database(
    name: NameString,
    description: Option[DescriptionString] = None,
    parameters: Option[Map[KeyString, ParametersMapValue]] = None
) {
  def buildAwsValue(): software.amazon.awssdk.services.athena.model.Database = {
    import Database.zioAwsBuilderHelper.BuilderOps
    software.amazon.awssdk.services.athena.model.Database
      .builder()
      .name(NameString.unwrap(name): java.lang.String)
      .optionallyWith(
        description.map(value =>
          DescriptionString.unwrap(value): java.lang.String
        )
      )(_.description)
      .optionallyWith(
        parameters.map(value =>
          value
            .map({ case (key, value) =>
              (KeyString.unwrap(key): java.lang.String) -> (ParametersMapValue
                .unwrap(value): java.lang.String)
            })
            .asJava
        )
      )(_.parameters)
      .build()
  }
  def asReadOnly: zio.aws.athena.model.Database.ReadOnly =
    zio.aws.athena.model.Database.wrap(buildAwsValue())
}
object Database {
  private lazy val zioAwsBuilderHelper
      : BuilderHelper[software.amazon.awssdk.services.athena.model.Database] =
    BuilderHelper.apply
  trait ReadOnly {
    def asEditable: zio.aws.athena.model.Database =
      zio.aws.athena.model.Database(
        name,
        description.map(value => value),
        parameters.map(value => value)
      )
    def name: NameString
    def description: Option[DescriptionString]
    def parameters: Option[Map[KeyString, ParametersMapValue]]
    def getName: ZIO[Any, Nothing, NameString] = ZIO.succeed(name)
    def getDescription: ZIO[Any, AwsError, DescriptionString] =
      AwsError.unwrapOptionField("description", description)
    def getParameters: ZIO[Any, AwsError, Map[KeyString, ParametersMapValue]] =
      AwsError.unwrapOptionField("parameters", parameters)
  }
  private final class Wrapper(
      impl: software.amazon.awssdk.services.athena.model.Database
  ) extends zio.aws.athena.model.Database.ReadOnly {
    override val name: NameString =
      zio.aws.athena.model.primitives.NameString(impl.name())
    override val description: Option[DescriptionString] = scala
      .Option(impl.description())
      .map(value => zio.aws.athena.model.primitives.DescriptionString(value))
    override val parameters: Option[Map[KeyString, ParametersMapValue]] = scala
      .Option(impl.parameters())
      .map(value =>
        value.asScala
          .map({ case (key, value) =>
            zio.aws.athena.model.primitives.KeyString(
              key
            ) -> zio.aws.athena.model.primitives.ParametersMapValue(value)
          })
          .toMap
      )
  }
  def wrap(
      impl: software.amazon.awssdk.services.athena.model.Database
  ): zio.aws.athena.model.Database.ReadOnly = new Wrapper(impl)
}
