/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.gsheetjdbc.loader;

import com.yahoo.gsheetjdbc.schema.Column;
import com.yahoo.gsheetjdbc.schema.Table;
import java.io.Closeable;
import java.io.IOException;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.List;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DatabaseLoader
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(DatabaseLoader.class);
    private static final String JDBC_DRIVER = "org.h2.Driver";
    private final String dbName;
    private Connection connection;
    private final String jdbcUrl;

    public DatabaseLoader(String dbName) {
        this.dbName = dbName;
        this.jdbcUrl = this.getH2URL();
    }

    public Connection getConnection() throws SQLException {
        return DriverManager.getConnection(this.jdbcUrl, "", "");
    }

    String getH2URL() {
        return String.format("jdbc:h2:mem:%s;DB_CLOSE_DELAY=-1", this.dbName);
    }

    void createAndLoadTable(Table table, List<List<Object>> records, String tableSuffix) throws SQLException {
        this.executeStatement(this.generateTableCreationStatement(table, tableSuffix));
        for (List<Object> record : records) {
            this.executeStatement(this.generateTableInsertionStatement(table, tableSuffix), record);
        }
    }

    public void refreshTempTable(Table table, List<List<Object>> records) throws SQLException {
        this.executeStatement(this.generateSchemaGenerationStatement(table));
        this.executeStatement(this.generateTableDropStatement(table, "Temp"));
        this.createAndLoadTable(table, records, "Temp");
    }

    public synchronized void swapTables(Table table) throws SQLException {
        this.executeStatement(this.generateTableRenameStatement(table, "", "Old"));
        this.executeStatement(this.generateTableRenameStatement(table, "Temp", ""));
        this.executeStatement(this.generateTableDropStatement(table, "Old"));
    }

    String generateTableName(Table table, String suffix) {
        StringBuilder statement = new StringBuilder();
        statement.append("\"");
        statement.append(table.getSchema());
        statement.append("\".\"");
        statement.append(table.getTableName());
        if (suffix != null) {
            statement.append(suffix);
        }
        statement.append("\"");
        return statement.toString();
    }

    String generateTableRenameStatement(Table table, String fromSuffix, String toSuffix) {
        StringBuilder statement = new StringBuilder();
        statement.append("ALTER TABLE IF EXISTS ");
        statement.append(this.generateTableName(table, fromSuffix));
        statement.append(" RENAME TO ");
        statement.append(this.generateTableName(table, toSuffix));
        return statement.toString();
    }

    String generateTableDropStatement(Table table, String tableSuffix) {
        StringBuilder statement = new StringBuilder();
        statement.append("DROP TABLE IF EXISTS ");
        statement.append(this.generateTableName(table, tableSuffix));
        return statement.toString();
    }

    String generateSchemaGenerationStatement(Table table) {
        StringBuilder statement = new StringBuilder();
        statement.append("CREATE SCHEMA IF NOT EXISTS \"");
        statement.append(table.getSchema());
        statement.append("\";");
        return statement.toString();
    }

    String generateTableCreationStatement(Table table, String tableSuffix) {
        StringBuilder statement = new StringBuilder();
        statement.append("CREATE TABLE IF NOT EXISTS ");
        statement.append(this.generateTableName(table, tableSuffix));
        statement.append(" (");
        statement.append(table.getColumns().stream().map(column -> "\"" + column.getName() + "\" " + this.getH2Type(column.getType())).collect(Collectors.joining(",")));
        statement.append(");");
        return statement.toString();
    }

    String generateTableInsertionStatement(Table table, String tableSuffix) {
        StringBuilder statement = new StringBuilder();
        statement.append("INSERT INTO ");
        statement.append(this.generateTableName(table, tableSuffix == null ? "" : tableSuffix));
        statement.append(" (");
        statement.append(table.getColumns().stream().map(column -> "\"" + column.getName() + "\"").collect(Collectors.joining(",")));
        statement.append(") VALUES (");
        statement.append(table.getColumns().stream().map(c -> "?").collect(Collectors.joining(",")));
        statement.append(");");
        return statement.toString();
    }

    String getH2Type(Column.ColumnType columnType) {
        switch (columnType) {
            case DATE: {
                return "DATE";
            }
            case NUMBER: {
                return "DOUBLE";
            }
            case DATETIME: {
                return "TIMESTAMP";
            }
            case STRING: {
                return "VARCHAR";
            }
            case BOOLEAN: {
                return "BOOLEAN";
            }
        }
        return "VARCHAR";
    }

    private void executeStatement(String sql) throws SQLException {
        this.executeStatement(sql, List.of());
    }

    private void executeStatement(String sql, List<Object> arguments) throws SQLException {
        if (this.connection == null || this.connection.isClosed()) {
            this.connection = this.getConnection();
        }
        try (PreparedStatement statement = this.connection.prepareStatement(sql);){
            int idx = 1;
            for (Object obj : arguments) {
                statement.setObject(idx, obj);
                ++idx;
            }
            long start = System.currentTimeMillis();
            statement.execute();
            long end = System.currentTimeMillis();
            log.debug("Executed SQL: {} Runtime: {}ms", (Object)sql, (Object)(end - start));
        }
    }

    @Override
    public void close() throws IOException {
        try {
            this.executeStatement("SHUTDOWN");
            this.connection.close();
        }
        catch (SQLException e) {
            throw new IOException(e.getMessage());
        }
    }
}

