package dev.speakeasyapi.sdk.client;

import dev.speakeasyapi.sdk.Ingest;
import dev.speakeasyapi.sdk.IngestServiceGrpc;
import io.grpc.ManagedChannel;
import io.grpc.ManagedChannelBuilder;
import io.grpc.Metadata;
import io.grpc.StatusRuntimeException;
import io.grpc.netty.shaded.io.grpc.netty.GrpcSslContexts;
import io.grpc.netty.shaded.io.grpc.netty.NettyChannelBuilder;
import io.grpc.netty.shaded.io.netty.handler.ssl.util.InsecureTrustManagerFactory;
import io.grpc.stub.MetadataUtils;
import org.springframework.util.StringUtils;

import javax.net.ssl.SSLException;

import static io.grpc.Metadata.ASCII_STRING_MARSHALLER;

public class SpeakeasyClient implements ISpeakeasyClient {
    private String authKey = "x-api-key";
    private Metadata.Key<String> metadataKey = Metadata.Key.of(authKey, ASCII_STRING_MARSHALLER);
    private String apiKey;
    private String serverUrl;
    private boolean secureGrpc;

    public SpeakeasyClient(String apiKey, String serverUrl, boolean secureGrpc) throws IllegalArgumentException {
        if (!StringUtils.hasText(apiKey)) {
            throw new IllegalArgumentException("Speakeasy API key is required.");
        }
        this.apiKey = apiKey;
        this.serverUrl = serverUrl;
        this.secureGrpc = secureGrpc;
    }

    @Override
    public void ingestGrpc(String harString) throws StatusRuntimeException, SSLException {
        Metadata metadata = new Metadata();
        metadata.put(metadataKey, apiKey);

        ManagedChannelBuilder channelBuilder;
        if (secureGrpc) {
            // Enable tls with self-signed cert
            channelBuilder = NettyChannelBuilder.forTarget(this.serverUrl)
                    .sslContext(GrpcSslContexts.forClient()
                            .trustManager(InsecureTrustManagerFactory.INSTANCE)
                            .build());
        } else {
            channelBuilder = ManagedChannelBuilder
                    .forTarget(this.serverUrl)
                    .usePlaintext();
        }

        ManagedChannel channel = channelBuilder
                .intercept(MetadataUtils.newAttachHeadersInterceptor(metadata))
                .build();

        IngestServiceGrpc.IngestServiceBlockingStub blockingStub = IngestServiceGrpc.newBlockingStub(channel);

        Ingest.IngestRequest ingestRequest = Ingest.IngestRequest.newBuilder()
                .setHar(harString)
                .build();
        try {
            blockingStub.ingest(ingestRequest);
        } catch (StatusRuntimeException e) {
            throw e;
        } finally {
            channel.shutdown();
        }
    }
}

