package dev.speakeasyapi.sdk;

import java.io.IOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletResponse;

import org.springframework.http.HttpStatus;
import org.springframework.util.StringUtils;
import org.springframework.web.util.ContentCachingRequestWrapper;
import org.springframework.web.util.ContentCachingResponseWrapper;

import com.smartbear.har.builder.HarContentBuilder;
import com.smartbear.har.builder.HarCookieBuilder;
import com.smartbear.har.builder.HarEntryBuilder;
import com.smartbear.har.builder.HarHeaderBuilder;
import com.smartbear.har.builder.HarPostDataBuilder;
import com.smartbear.har.builder.HarRequestBuilder;
import com.smartbear.har.builder.HarResponseBuilder;
import com.smartbear.har.creator.DefaultHarStreamWriter;
import com.smartbear.har.model.HarContent;
import com.smartbear.har.model.HarCookie;
import com.smartbear.har.model.HarCreator;
import com.smartbear.har.model.HarHeader;
import com.smartbear.har.model.HarPostData;
import com.smartbear.har.model.HarRequest;
import com.smartbear.har.model.HarResponse;

public class SpeakeasyHarBuilder {
    private final String cookieResponseHeaderName = "Set-Cookie";

    private final DefaultHarStreamWriter.Builder harWriterBuilder;

    private OutputStream outputStream;

    public OutputStream getOutputStream() {
        return this.outputStream;
    }

    private String comment;

    public String getComment() {
        return this.comment;
    }

    private Date startTime = new Date();

    public Date getStartTime() {
        return this.startTime;
    }

    private Date endTime = new Date();

    public Date getEndTime() {
        return this.endTime;
    }

    private String hostName;

    public String getHostName() {
        return this.hostName;
    }

    private String port;

    public String getPort() {
        return this.port;
    }

    private HarCreator creator;

    public HarCreator getCreator() {
        return this.creator;
    }

    private HarRequest harRequest;

    public HarRequest getHarRequest() {
        return this.harRequest;
    }

    private HarResponse harResponse;

    public HarResponse getHarResponse() {
        return this.harResponse;
    }

    public SpeakeasyHarBuilder() {
        this.harWriterBuilder = new DefaultHarStreamWriter.Builder();
    }

    public SpeakeasyHarBuilder withStartTime(Date startTime) {
        this.startTime = startTime;
        return this;
    }

    public SpeakeasyHarBuilder withEndTime(Date endTime) {
        this.endTime = endTime;
        return this;
    }

    public SpeakeasyHarBuilder withHostName(String hostName) {
        this.hostName = hostName;
        return this;
    }

    public SpeakeasyHarBuilder withPort(int port) {
        this.port = String.valueOf(port);
        return this;
    }

    public SpeakeasyHarBuilder withRequest(ContentCachingRequestWrapper request)
            throws IOException {
        Map<String, List<String>> headerMap = Collections.list(request.getHeaderNames()).stream()
                .collect(Collectors.toMap(h -> h, h -> Collections.list(request.getHeaders(h))));

        List<HarCookie> harCookieList = new ArrayList<>();
        // Note: It appears that java inserts cookies with headers,
        // so this never gets called.
        if (request.getCookies() != null) {
            // Parse cookies
            harCookieList = Arrays.stream(request.getCookies()).map(c -> {
                return new HarCookieBuilder()
                        .withName(c.getName())
                        .withValue(c.getValue()).build();
            }).collect(Collectors.toList());
        }

        // Parse headers
        List<HarHeader> harHeaderList = new ArrayList<>();
        Enumeration<String> headerNames = request.getHeaderNames();
        while (headerNames.hasMoreElements()) {
            String name = headerNames.nextElement();
            harHeaderList.add(new HarHeaderBuilder()
                    .withName(name)
                    .withValues(Collections.list(request.getHeaders(name)))
                    .build());
        }

        HarPostData postData = null;

        if (request.getContentLength() > 0) {
            if (request.getContentAsByteArray().length == 0) {
                // request not read, we will read it to get the content
                request.getReader().lines().collect(Collectors.toList());
            }

            postData = new HarPostDataBuilder()
                    .withMimeType(request.getContentType())
                    .withText(new String(request.getContentAsByteArray()))
                    .build();
        }

        String queryString = StringUtils.hasText(request.getQueryString()) ? request.getQueryString() : "";

        HarRequestBuilder builder = new HarRequestBuilder()
                .withBodySize(request.getContentLengthLong())
                .withCookies(harCookieList)
                .withHeaders(harHeaderList)
                .withHeadersSize(calculateHeaderSize(headerMap))
                .withHttpVersion(request.getProtocol())
                .withMethod(request.getMethod())
                .withQueryString(queryString)
                .withUrl(request.getRequestURI());

        if (postData != null) {
            builder.withPostData(postData);
        }

        this.harRequest = builder.build();
        return this;
    }

    public SpeakeasyHarBuilder withResponse(ContentCachingResponseWrapper response, String httpVersion) {
        Map<String, List<String>> headerMap = response.getHeaderNames().stream()
                .collect(Collectors.toMap(h -> h, h -> new ArrayList<>(response.getHeaders(h))));

        // Parse cookies
        List<HarCookie> harCookieList = response.getHeaderNames()
                .stream()
                .filter(header -> cookieResponseHeaderName.equals(header))
                .flatMap(headerName -> response.getHeaders(headerName).stream().map(hv -> parseSetCookieString(hv)))
                .collect(Collectors.toList());

        // Parse headers
        List<HarHeader> harHeaderList = response.getHeaderNames()
                .stream()
                .map(headerName -> new HarHeaderBuilder()
                        .withName(headerName)
                        .withValues(new ArrayList<>(response.getHeaders(headerName)))
                        .build())
                .collect(Collectors.toList());

        String contentType = response.getContentType();
        if (!StringUtils.hasText(contentType)) {
            contentType = "application/octet-stream"; // Default HTTP content type
        }

        HarContent harContent = new HarContentBuilder()
                .withSize((long) response.getContentSize())
                .withText(new String(response.getContentAsByteArray()))
                .withMimeType(contentType)
                .build();

        String redirectURL = "";
        Collection<String> locationHeaders = response.getHeaders("Location");
        if (locationHeaders.size() > 0) {
            redirectURL = locationHeaders.iterator().next();
        }

        long bodySize = response.getContentSize();
        if (response.getStatus() == HttpServletResponse.SC_NOT_MODIFIED) {
            bodySize = 0;
        }

        this.harResponse = new HarResponseBuilder()
                .withStatus(response.getStatus())
                .withStatusText(HttpStatus.valueOf(response.getStatus()).getReasonPhrase())
                .withCookies(harCookieList)
                .withContent(harContent)
                .withBodySize(bodySize)
                .withHeaders(harHeaderList)
                .withHeadersSize(calculateHeaderSize(headerMap))
                .withHttpVersion(httpVersion)
                .withRedirectURL(redirectURL)
                .build();

        return this;
    }

    public SpeakeasyHarBuilder withOutputStream(OutputStream outputStream) {
        this.outputStream = outputStream;
        return this;
    }

    public SpeakeasyHarBuilder withComment(String comment) {
        this.comment = comment;
        return this;
    }

    public SpeakeasyHarBuilder withCreator(HarCreator creator) {
        this.creator = creator;
        return this;
    }

    private HarCookie parseSetCookieString(String cookie) {
        HarCookieBuilder cookieBuilder = new HarCookieBuilder();

        String[] cookieParts = cookie.split("; ");

        for (String cookiePart : cookieParts) {
            String cookieRegex = "([^=]+)=?([^=]*)";
            Pattern pattern = Pattern.compile(cookieRegex);
            Matcher matcher = pattern.matcher(cookiePart);

            while (matcher.find()) {
                String key = matcher.group(1);
                String value = matcher.group(2);
                switch (key.toLowerCase()) {
                    case "domain":
                        cookieBuilder.withDomain(value);
                        break;
                    case "max-age": {
                        int maxAge = Integer.parseInt(value);

                        Calendar calendar = Calendar.getInstance();
                        calendar.setTime(startTime);
                        calendar.add(Calendar.SECOND, maxAge);

                        SimpleDateFormat outFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSXXX");
                        outFormat.setTimeZone(TimeZone.getTimeZone("UTC"));

                        String expires = outFormat.format(calendar.getTime());

                        cookieBuilder.withExpires(expires);
                        break;
                    }
                    case "expires": {
                        try {
                            SimpleDateFormat inFormat = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss zzz");
                            Date expiresDate = inFormat.parse(value);

                            SimpleDateFormat outFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSXXX");
                            outFormat.setTimeZone(TimeZone.getTimeZone("UTC"));

                            String expires = outFormat.format(expiresDate);

                            cookieBuilder.withExpires(expires);
                        } catch (Exception e) {
                        }
                        break;
                    }
                    case "httponly":
                        cookieBuilder.withHttpOnly(true);
                        break;
                    case "path":
                        cookieBuilder.withPath(value);
                        break;
                    case "secure":
                        cookieBuilder.withSecure(true);
                        break;
                    default:
                        cookieBuilder.withName(key);
                        cookieBuilder.withValue(value);
                        break;
                }
            }
        }
        return cookieBuilder.build();
    }

    public void build() throws IOException {
        DefaultHarStreamWriter harWriter = harWriterBuilder
                .withOutputStream(outputStream)
                .withComment(comment)
                .withCreator(creator)
                .build();

        HarEntryBuilder builder = new HarEntryBuilder()
                .withRequest(harRequest)
                .withResponse(harResponse)
                .withServerIPAddress(hostName)
                .withStartedDateTime(startTime)
                .withTime(endTime.getTime() - startTime.getTime())
                .withCache(null)
                .withTimings(null);

        if (!port.equals("-1")) {
            builder.withConnection(port);
        }

        harWriter.addEntry(builder.build());
        harWriter.closeHar();
    }

    private long calculateHeaderSize(Map<String, List<String>> headers) {
        StringBuilder builder = new StringBuilder();

        headers.forEach((key, values) -> {
            for (String value : values) {
                builder.append(key)
                        .append(": ")
                        .append(value)
                        .append("\r\n");
            }
        });

        return builder.toString().getBytes().length;
    }
}
