package dev.speakeasyapi.sdk;

import com.smartbear.har.builder.*;
import com.smartbear.har.creator.DefaultHarStreamWriter;
import com.smartbear.har.model.*;
import org.springframework.http.HttpStatus;
import org.springframework.util.StringUtils;
import org.springframework.web.util.ContentCachingRequestWrapper;
import org.springframework.web.util.ContentCachingResponseWrapper;

import java.io.IOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

public class SpeakeasyHarBuilder {
    private final String cookieResponseHeaderName = "Set-Cookie";

    private final DefaultHarStreamWriter.Builder harWriterBuilder;

    private OutputStream outputStream;

    public OutputStream getOutputStream() {
        return this.outputStream;
    }

    private String comment;

    public String getComment() {
        return this.comment;
    }

    private Date startTime;

    public Date getStartTime() {
        return this.startTime;
    }

    private String hostName;

    public String getHostName() {
        return this.hostName;
    }

    private String port;

    public String getPort() {
        return this.port;
    }

    private HarCreator creator;

    public HarCreator getCreator() {
        return this.creator;
    }

    private HarRequest harRequest;

    public HarRequest getHarRequest() {
        return this.harRequest;
    }

    private HarResponse harResponse;

    public HarResponse getHarResponse() {
        return this.harResponse;
    }

    public SpeakeasyHarBuilder() {
        this.harWriterBuilder = new DefaultHarStreamWriter.Builder();
    }

    public SpeakeasyHarBuilder withStartTime(Date startTime) {
        this.startTime = startTime;
        return this;
    }

    public SpeakeasyHarBuilder withHostName(String hostName) {
        this.hostName = hostName;
        return this;
    }

    public SpeakeasyHarBuilder withPort(int port) {
        this.port = String.valueOf(port);
        return this;
    }

    public SpeakeasyHarBuilder withRequest(ContentCachingRequestWrapper request) throws UnsupportedEncodingException {

        List<HarCookie> harCookieList = new ArrayList<>();
        // Note: It appears that java inserts cookies with headers,
        // so this never gets called.
        if (request.getCookies() != null) {
            // Parse cookies
            harCookieList = Arrays.stream(request.getCookies()).map(c -> {
                HarCookie harCookie = new HarCookieBuilder()
                        .withComment(c.getComment())
                        .withDomain(c.getDomain())
                        .withExpires(String.valueOf(c.getMaxAge()))
                        .withHttpOnly(c.isHttpOnly())
                        .withName(c.getName())
                        .withPath(c.getPath())
                        .withSecure(c.getSecure())
                        .withValue(c.getValue())
                        .build();
                return harCookie;
            }).collect(Collectors.toList());
        }

        // Parse headers
        List<HarHeader> harHeaderList = new ArrayList<>();
        Enumeration<String> headerNames = request.getHeaderNames();
        while (headerNames.hasMoreElements()) {
            String name = headerNames.nextElement();
            String cookieRequestHeaderName = "Cookie";
            if (name.equalsIgnoreCase(cookieRequestHeaderName)) {
                Enumeration<String> cookieValues = request.getHeaders(name);
                while (cookieValues.hasMoreElements()) {
                    harCookieList.add(parseCookieString(cookieValues.nextElement()));
                }
            } else {
                harHeaderList.add(new HarHeaderBuilder()
                        .withName(name)
                        .withValues(Collections.list(request.getHeaders(name)))
                        .build());
            }
        }

        String queryString = StringUtils.hasText(request.getQueryString()) ? request.getQueryString() : "";
        this.harRequest = new HarRequestBuilder()
                .withPostData(new HarPostDataBuilder()
                        .withMimeType(request.getContentType())
                        .withText(new String(request.getContentAsByteArray()))
                        .build()
                )
                .withBodySize(request.getContentLengthLong())
                .withCookies(harCookieList)
                .withHeaders(harHeaderList)
                .withHeadersSize(-1l)       // TODO do we need to calculate this? If so we can get it with r.Header.Write to a bytes.Buffer and read size
                .withHttpVersion(request.getProtocol())
                .withMethod(request.getMethod())
                .withQueryString(queryString)
                .withUrl(request.getRequestURI())
                .build();
        return this;
    }

    public SpeakeasyHarBuilder withResponse(ContentCachingResponseWrapper response) {
        // Parse cookies
        List<HarCookie> harCookieList = response.getHeaderNames()
                .stream()
                .filter(header -> cookieResponseHeaderName.equals(header))
                .flatMap(headerName -> response.getHeaders(headerName).stream().map(hv -> parseCookieString(hv)))
                .collect(Collectors.toList());

        // Parse headers
        List<HarHeader> harHeaderList = response.getHeaderNames()
                .stream()
                .filter(header -> !cookieResponseHeaderName.equals(header))
                .map(headerName -> new HarHeaderBuilder()
                        .withName(headerName)
                        .withValues(new ArrayList<>(response.getHeaders(headerName)))
                        .build()
                )
                .collect(Collectors.toList());

        HarContent harContent = new HarContentBuilder()
                .withSize((long) response.getContentSize())
                .withText(new String(response.getContentAsByteArray()))
                .withMimeType(response.getContentType())
                .build();

        this.harResponse = new HarResponseBuilder()
                .withStatus(response.getStatus())
                .withStatusText(HttpStatus.valueOf(response.getStatus()).name())
                .withCookies(harCookieList)
                .withContent(harContent)
                .withBodySize((long) response.getContentSize())
                .withHeaders(harHeaderList)
                .withHeadersSize(-1l)
                .build();
        return this;
    }

    public SpeakeasyHarBuilder withOutputStream(OutputStream outputStream) {
        this.outputStream = outputStream;
        return this;
    }

    public SpeakeasyHarBuilder withComment(String comment) {
        this.comment = comment;
        return this;
    }

    public SpeakeasyHarBuilder withCreator(HarCreator creator) {
        this.creator = creator;
        return this;
    }

    private HarCookie parseCookieString(String cookie) {
        String cookieHeaderPrefix = "Set-Cookie=";
        if (cookie.startsWith(cookieHeaderPrefix)) {
            cookie = cookie.substring(cookieHeaderPrefix.length());
        }
        HarCookieBuilder cookieBuilder = new HarCookieBuilder();
        String cookieRegex = "([^=]+)=([^\\;]+);\\s?";
        Pattern pattern = Pattern.compile(cookieRegex);
        Matcher matcher = pattern.matcher(cookie);

        while (matcher.find()) {
            String key = matcher.group(1);
            String value = matcher.group(2);
            switch (key.toLowerCase()) {
                case "domain":
                    cookieBuilder.withDomain(value);
                    break;
                case "expires":
                    cookieBuilder.withExpires(value);
                    break;
                case "httponly":
                    cookieBuilder.withHttpOnly("true".equalsIgnoreCase(value));
                    break;
                case "path":
                    cookieBuilder.withPath(value);
                    break;
                case "secure":
                    cookieBuilder.withSecure("true".equalsIgnoreCase(value));
                    break;
                default:
                    cookieBuilder.withName(key);
                    cookieBuilder.withValue(value);
                    break;
            }
        }
        return cookieBuilder.build();
    }

    public void build() throws IOException {
        DefaultHarStreamWriter harWriter = harWriterBuilder
                .withOutputStream(outputStream)
                .withComment(comment)
                .withCreator(creator)
                .build();


        harWriter.addEntry(new HarEntryBuilder()
                .withConnection(port)
                .withRequest(harRequest)
                .withResponse(harResponse)
                .withServerIPAddress(hostName)
                .withStartedDateTime(startTime)
                .withTime(new Date().getTime())
                .build()
        );
        harWriter.closeHar();
    }
}
