/**
 * Copyright 2020 SkillTree
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package callStack.utils;

import org.apache.commons.lang3.Validate;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicInteger;

public abstract class AbstractThreadPool implements ThreadPoolStats {

	protected ExecutorService m_pool = null;

	public List<String> submitAndGetExceptions(Callable<?> submitMe,
			int numOfSubmissions) {
		List<String> allExceptions = submitAndGetExceptions(m_pool, submitMe,
				numOfSubmissions);
		return allExceptions;
	}

	public <T> List<String> submitAndGetExceptions(List<Callable<T>> submitList) {
		List<String> allExceptions = submitAndGetExceptions(m_pool, submitList);
		return allExceptions;
	}

	public <T> List<Future<T>> submit(Callable<T> submitMe, int numOfSubmissions) {
		return submit(m_pool, submitMe, numOfSubmissions);
	}

	public <T> List<Future<T>> submit(List<Callable<T>> listToSubmit) {
		return submitAll(m_pool, listToSubmit);
	}

	public <T> List<T> submitAndGetResults(List<Callable<T>> listToSubmit) {
		List<Future<T>> futures = submitAll(m_pool, listToSubmit);

		List<T> results = pullOutResults(futures);
		return results;
	}

	private <T> List<T> pullOutResults(List<Future<T>> futures) {
		List<T> results = new ArrayList<T>();
		for (Future<T> future : futures) {
			try {
				T result = future.get();
				if (result != null) {
					results.add(result);
				}
			} catch (Throwable e) {
			    throw new RuntimeException("Failed to execute callable.", e);
            }
		}
		return results;
	}

	public <T> List<T> submitAndGetResults(Callable<T> callable,
			int numOfSubmissions) {
		Validate.notNull(callable);
		Validate.isTrue(numOfSubmissions > 0,
				"Must at submit at least 1 callable");

		List<Future<T>> futures = submit(callable, numOfSubmissions);
		return pullOutResults(futures);
	}

	public void shutdown() {
		m_pool.shutdown();
	}

	/**
	 * The default thread factory
	 *
	 * Code borrowed from JDK, named was added to the constructor
	 */
	static class NamedThreadFactory implements ThreadFactory {
		final ThreadGroup group;

		final AtomicInteger threadNumber = new AtomicInteger(1);

		final String namePrefix;

		NamedThreadFactory(String name) {
			SecurityManager s = System.getSecurityManager();
			group = (s != null) ? s.getThreadGroup() : Thread.currentThread()
					.getThreadGroup();
			namePrefix = name + "-";
		}

		@Override
		public Thread newThread(Runnable r) {
			Thread t = new Thread(group, r, namePrefix
					+ threadNumber.getAndIncrement(), 0);
			if (t.isDaemon())
				t.setDaemon(false);
			if (t.getPriority() != Thread.NORM_PRIORITY)
				t.setPriority(Thread.NORM_PRIORITY);
			return t;
		}
	}

	/**
	 * Submit provided {@link Callable} numOfSubmissions times, wait for ALL the
	 * threads to finish and return a list of the exceptions generated by the
	 * threads. If there are no exceptions then an empty list is returned.
	 *
	 * @param executor
	 *            - executor service to use
	 * @param callable
	 *            - {@link Callable} to submit numOfSubmissions times
	 * @param numOfSubmissions
	 *            number of times to submit the provided callable
	 * @return
	 */
	private <T extends Object> List<String> submitAndGetExceptions(
			final ExecutorService executor, final Callable<T> callable,
			final int numOfSubmissions) {
		List<Future<T>> futures = submit(executor, callable, numOfSubmissions);
		return getExceptionsInList(futures);
	}

	/**
	 * Submit provided {@link Callable} numOfSubmissions times and return the
	 * futures
	 *
	 * @param executor
	 *            - executor service to use
	 * @param callable
	 *            - {@link Callable} to submit numOfSubmissions times
	 * @param numOfSubmissions
	 *            number of times to submit the provided callable
	 * @return a list of futures
	 */
	private <T extends Object> List<Future<T>> submit(
			final ExecutorService executor, final Callable<T> callable,
			final int numOfSubmissions) {
		List<Future<T>> futures = new ArrayList<Future<T>>(numOfSubmissions);
		for (int i = 0; i < numOfSubmissions; i++) {
			futures.add(executor.submit(callable));
		}

		return futures;
	}

	/**
	 * Retrieves all Exceptions from the List of Future objects and puts them in
	 * a list of Strings. Note: will halt the calling thread till all the
	 * futures/threads are completed.
	 *
	 * @param futures
	 * @return the List of Strings, one element for each Future's Exception
	 */
	private <T extends Object> List<String> getExceptionsInList(
			final List<Future<T>> futures) {
		List<String> exceptions = new ArrayList<String>(futures.size());

		for (Future<T> future : futures) {
			try {
				future.get();
			} catch (Throwable append) {
				exceptions.add(getStackTraceFromThrowable(append));
			}
		}

		return exceptions;
	}

	private String getStackTraceFromThrowable(Throwable t) {
		if (t != null) {
			StringWriter writer = new StringWriter();
			PrintWriter printWriter = new PrintWriter(writer);
			t.printStackTrace(printWriter);
			String stackTrace = writer.toString();
			printWriter.close();
			return stackTrace;
		} else {
			return null;
		}
	}

	/**
	 * Submit a list of {@link Callable}s, wait for them to execute and return
	 * exceptions
	 *
	 * @param executor
	 *            - executor service to use
	 * @param callables
	 *            - a list o {@link Callable}s to submit
	 * @return
	 */
	private <T extends Object> List<String> submitAndGetExceptions(
			final ExecutorService executor, final List<Callable<T>> callables) {
		List<Future<T>> futures = submitAll(executor, callables);
		return getExceptionsInList(futures);
	}

	/**
	 * Submits all {@link Callable} tasks in the list using the provided
	 * ExecutorService
	 *
	 * @param <T>
	 * @param executor
	 * @param callables
	 * @return a list of Future objects, one for each submitted task
	 */
	private <T extends Object> List<Future<T>> submitAll(
			final ExecutorService executor, final List<Callable<T>> callables) {
		List<Future<T>> futures = new ArrayList<Future<T>>(callables.size());
		for (Callable<T> callable : callables) {
			futures.add(executor.submit(callable));
		}
		return futures;
	}

	/**
	 * Retrieves all Exceptions from the List of Future objects and appends them
	 * to a String. Note: will halt the calling thread till all the
	 * futures/threads are completed.
	 *
	 * @param futures list of futures to invoke
	 * @return a String containing all Exceptions thrown from the Future tasks
	 */
	public <T extends Object> String getExceptions(final List<Future<T>> futures) {
		StringBuilder builder = new StringBuilder();
		for (Future<T> future : futures) {
			try {
				future.get();
			} catch (Throwable append) {
				builder.append(getStackTraceFromThrowable(append));
			}
		}

		return builder.toString();
	}

	/**
	 * (U) Returns the maximum size of this thread pool. Some implementations of
	 * the underlying Executor may not expose the thread pool sizes. If this
	 * occurs, zero will be returned.
	 *
	 * @return The maximum size of this thread pool.
	 */
	public int getMaximumPoolSize() {
		if (m_pool instanceof ThreadPoolExecutor) {
			return ((ThreadPoolExecutor) m_pool).getMaximumPoolSize();
		} else {
			return 0;
		}
	}

	/**
	 * (U) Returns the current size of this thread pool. Some implementations of
	 * the underlying Executor may not expose the thread pool sizes. If this
	 * occurs, zero will be returned.
	 *
	 * @return The current size of this thread pool.
	 */
	public int getCurrentPoolSize() {
		if (m_pool instanceof ThreadPoolExecutor) {
			return ((ThreadPoolExecutor) m_pool).getPoolSize();
		} else {
			return 0;
		}
	}

	/**
	 * (U) Returns the number of actively running threads in this thread pool.
	 * Some implementations of the underlying Executor may not expose the thread
	 * pool sizes. If this occurs, zero will be returned.
	 *
	 * @return The active size of this thread pool.
	 */
	public int getActivePoolSize() {
		if (m_pool instanceof ThreadPoolExecutor) {
			return ((ThreadPoolExecutor) m_pool).getActiveCount();
		} else {
			return 0;
		}
	}
}
