// Copyright (c) 2023 - Restate Software, Inc., Restate GmbH
//
// This file is part of the Restate Java SDK,
// which is released under the MIT license.
//
// You can find a copy of the license in file LICENSE in the root
// directory of this repository or package, or at
// https://github.com/restatedev/sdk-java/blob/main/LICENSE
package dev.restate.sdk.common;

import com.google.protobuf.ByteString;
import com.google.protobuf.UnsafeByteOperations;
import dev.restate.sdk.common.function.ThrowingFunction;
import java.util.Objects;
import javax.annotation.Nullable;

/** Interface defining serialization and deserialization of concrete types. */
public interface Serde<T> {

  byte[] serialize(@Nullable T value);

  default ByteString serializeToByteString(@Nullable T value) {
    // This is safe because we don't mutate the generated byte[] afterward.
    return UnsafeByteOperations.unsafeWrap(serialize(value));
  }

  T deserialize(byte[] value);

  default T deserialize(ByteString byteString) {
    return deserialize(byteString.toByteArray());
  }

  /**
   * Create a {@link Serde} from {@code serializer}/{@code deserializer} lambdas. Before invoking
   * the serializer, we check that {@code value} is non-null.
   */
  static <T> Serde<T> using(
      ThrowingFunction<T, byte[]> serializer, ThrowingFunction<byte[], T> deserializer) {
    return new Serde<>() {
      @Override
      public byte[] serialize(T value) {
        return serializer.asFunction().apply(Objects.requireNonNull(value));
      }

      @Override
      public T deserialize(byte[] value) {
        return deserializer.asFunction().apply(value);
      }
    };
  }
}
