package dev.openmobile.ui.utils

import androidx.compose.foundation.*
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.composed
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.unit.dp
import androidx.core.graphics.toColorInt
import dev.openmobile.ui.UIManager

fun Arrangement.verticalArrangementFromJson(arrangement: String): Arrangement.Vertical {
    return when (arrangement) {
        "top" -> Top
        "center" -> Center
        "bottom" -> Bottom
        "space_around" -> SpaceAround
        "space_between" -> SpaceBetween
        "space_evenly" -> SpaceEvenly
        else -> Top
    }
}

fun Arrangement.horizontalArrangementFromJson(arrangement: String): Arrangement.Horizontal {
    return when (arrangement) {
        "start" -> Start
        "center" -> Center
        "end" -> End
        "space_around" -> SpaceAround
        "space_between" -> SpaceBetween
        "space_evenly" -> SpaceEvenly
        else -> Start
    }
}

fun Alignment.Companion.horizontalAlignmentFromJson(arrangement: String): Alignment.Horizontal {
    return when (arrangement) {
        "start" -> Start
        "center" -> CenterHorizontally
        "end" -> End
        else -> Start
    }
}

fun Alignment.Companion.verticalAlignmentFromJson(arrangement: String): Alignment.Vertical {
    return when (arrangement) {
        "top" -> Top
        "center" -> CenterVertically
        "end" -> Bottom
        else -> Top
    }
}

fun ContentScale.Companion.fromString(scale: String): ContentScale {
    return when (scale) {
        "fit" -> Fit
        "fill" -> FillBounds
        else -> Fit
    }
}

fun Any?.parseToDouble(default: Double? = null): Double? {
    if (this is Double?) return this ?: default
    if (this is Integer?) return this?.toDouble() ?: default
    return null
}

fun Modifier.parseFromJson(attributes: Map<*, *>?): Modifier = composed {
    attributes?.let {
        val width = attributes["width"].parseToDouble()?.dp
        val height = attributes["height"].parseToDouble()?.dp
        val fillMaxHeight = (attributes["fillMaxHeight"] as Boolean?) ?: false
        val fillMaxWidth = (attributes["fillMaxWidth"] as Boolean?) ?: false

        val padding = attributes["padding"] as Map<*, *>?
        val top = padding?.get("top").parseToDouble(0.0)!!.dp
        val start = padding?.get("start").parseToDouble(0.0)!!.dp
        val end = padding?.get("end").parseToDouble(0.0)!!.dp
        val bottom = padding?.get("bottom").parseToDouble(0.0)!!.dp

        val margin = attributes["margin"] as Map<*, *>?
        val mTop = margin?.get("top").parseToDouble(0.0)!!.dp
        val mStart = margin?.get("start").parseToDouble(0.0)!!.dp
        val mEnd = margin?.get("end").parseToDouble(0.0)!!.dp
        val mBottom = margin?.get("bottom").parseToDouble(0.0)!!.dp

        val color = Color((attributes["backgroundColor"] as String?)?.toColorInt() ?: Color.Unspecified.hashCode())
        val rounded = attributes["rounded"].parseToDouble(0.0)!!.dp
        val action = attributes["action"] as Map<*, *>?

        val verticalScroll = (attributes["verticalScroll"] as Boolean?) ?: false
        val horizontalScroll = (attributes["horizontalScroll"] as Boolean?) ?: false
        val verticalState = rememberScrollState()
        val horizontalState = rememberScrollState()

        padding(start = mStart, top = mTop, end = mEnd, bottom = mBottom)
            .background(color)
            .padding(start = start, top = top, end = end, bottom = bottom)
            .modifyIf(fillMaxHeight) { fillMaxHeight() }
            .modifyIf(fillMaxWidth) { fillMaxWidth() }
            .modifyIf(height != null && !fillMaxHeight) { height(height!!) }
            .modifyIf(width != null && !fillMaxWidth) { width(width!!) }
            .clip(RoundedCornerShape(rounded))
            .clickable(action != null) {
                UIManager.runActionByMap(action)
            }
            .modifyIf(verticalScroll) {
                verticalScroll(verticalState)
            }
            .modifyIf(horizontalScroll) {
                horizontalScroll(horizontalState)
            }
    } ?: run {
        Modifier
    }
}

fun Modifier.modifyIf(condition: Boolean, modify: Modifier.() -> Modifier) =
    if (condition) modify() else this
