package dev.openmobile.ui

import android.content.Context
import android.content.Intent
import android.os.Handler
import android.os.Looper
import android.util.Log
import android.widget.Toast
import androidx.compose.runtime.Composable
import dev.openmobile.core.ActionManager
import dev.openmobile.core.ActionType
import dev.openmobile.core.actions.UIActions
import dev.openmobile.ui.utils.UIParser

class ClientManager(private val context: Context) : ActionManager {

    override fun runAction(action: ActionType, parameters: Map<*, *>?): Any? {
        when (action) {
            UIActions.UI_LAUNCH_INTENT -> {
                parameters?.let {
                    Log.i("OpenMobile ", "UI_LAUNCH_INTENT parameters :: $parameters")
                    buildAndLaunchIntent(parameters["json"] as String)
                }
            }
            UIActions.UI_BUILD_INTENT -> {
                parameters?.let {
                    Log.i("OpenMobile ", "UI_BUILD_INTENT parameters :: $parameters")
                    return buildActionIntent(parameters)
                }
            }
            UIActions.UI_SHOW_TOAST -> {
                parameters?.let {
                    Log.i("OpenMobile ", "UI_SHOW_TOAST parameters :: $parameters")
                    showToast(parameters["message"])
                }
            }
            UIActions.UI_DISMISS_INTENT -> {
                dismissIntent()
            }
            UIActions.UI_OPEN_URL -> {
                parameters?.get("url")?.let { url ->
                    Log.i("OpenMobile ", "UI_OPEN_URL parameters :: $url")
                    openUrl(url as String)
                } ?: run {
                    Log.i("OpenMobile ", "Missing parameter url")
                }
            }
            else -> {}
        }
        return null
    }

    @Composable
    fun renderComponent(json: Map<*, *>) = UIParser(json = json)

    private fun buildAndLaunchIntent(json: String) {
        context.startActivity(buildLayoutIntent(json))
    }

    private fun buildLayoutIntent(json: String): Intent {
        val intent = Intent(context, LayoutActivity::class.java)
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
        intent.putExtra(LayoutActivity.LAYOUT_ACTIVITY_JSON_PARAM, json)
        return intent
    }

    private fun buildActionIntent(parameters: Map<*, *>): Intent? {
        parameters["action"]?.let { action ->
            Log.i("OpenMobile ", "action parameters :: $action")
            val intent = Intent(context, ActionIntentActivity::class.java)
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            intent.putExtra(ActionIntentActivity.INTENT_ACTIVITY_ACTION, action as String)
            return intent
        }
        return null
    }

    private fun showToast(message: Any?) {
        if (message == null || message !is String) {
            Handler(Looper.getMainLooper()).post {
                Toast.makeText(context, "Missing message attribute for Toast Message", Toast.LENGTH_SHORT).show()
            }
        } else {
            Handler(Looper.getMainLooper()).post {
                Toast.makeText(context, message, Toast.LENGTH_SHORT).show()
            }
        }
    }

    private fun openUrl(value: String) {
        val webview = "{\n" +
                "  \"type\": \"column\",\n" +
                "  \"attributes\": {\n" +
                "    \"modifier\": {\n" +
                "      \"backgroundColor\": \"#F7F4EA\",\n" +
                "      \"fillMaxHeight\": true,\n" +
                "      \"fillMaxWidth\": true\n" +
                "    },\n" +
                "    \"verticalArrangement\": \"top\",\n" +
                "    \"horizontalAlignment\": \"center\"\n" +
                "  },\n" +
                "  \"children\": [\n" +
                "    {\n" +
                "      \"type\": \"webView\",\n" +
                "      \"attributes\": {\n" +
                "        \"modifier\": {\n" +
                "          \"fillMaxHeight\": true,\n" +
                "          \"fillMaxWidth\": true\n" +
                "        },\n" +
                "        \"url\": \"$value\"\n" +
                "      }\n" +
                "    }\n" +
                "  ]\n" +
                "}\n"
        buildAndLaunchIntent(json = webview)
    }

    private fun dismissIntent() {
        Intent().also { intent ->
            intent.action = "dev.openmobile.ui.FINISH_ACTIVITY"
            UIManager.client.context.sendBroadcast(intent)
        }
    }
}