package dev.openmobile.ui.utils

import androidx.compose.foundation.background
import androidx.compose.foundation.layout.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Alignment.Companion.CenterHorizontally
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.unit.dp
import androidx.core.graphics.toColorInt

fun Arrangement.verticalArrangementFromJson(arrangement: String): Arrangement.Vertical {
    return when (arrangement) {
        "top" -> Top
        "center" -> Center
        "bottom" -> Bottom
        "space_around" -> SpaceAround
        "space_between" -> SpaceBetween
        "space_evenly" -> SpaceEvenly
        else -> Top
    }
}

fun Arrangement.horizontalArrangementFromJson(arrangement: String): Arrangement.Horizontal {
    return when (arrangement) {
        "start" -> Start
        "center" -> Center
        "end" -> End
        else -> Start
    }
}

fun Alignment.Companion.horizontalAlignmentFromJson(arrangement: String): Alignment.Horizontal {
    return when (arrangement) {
        "start" -> Start
        "center" -> CenterHorizontally
        "end" -> End
        else -> Start
    }
}

fun Alignment.Companion.verticalAlignmentFromJson(arrangement: String): Alignment.Vertical {
    return when (arrangement) {
        "top" -> Top
        "center" -> CenterVertically
        "end" -> Bottom
        else -> Top
    }
}

fun ContentScale.Companion.fromString(scale: String): ContentScale {
    return when (scale) {
        "fit" -> Fit
        "fill" -> FillBounds
        else -> Fit
    }
}

fun Modifier.parseFromJson(attributes: Map<*, *>?): Modifier {
    if (attributes.isNullOrEmpty()) return Modifier
    val width = (attributes["width"] as Double?)?.dp
    val height = (attributes["height"] as Double?)?.dp
    val fillMaxHeight = (attributes["fillMaxHeight"] as Boolean?) ?: false
    val fillMaxWidth = (attributes["fillMaxWidth"] as Boolean?) ?: false

    val padding = attributes["padding"] as Map<*, *>?
    val top = (padding?.get("top") as Double? ?: 0.0).dp
    val start = (padding?.get("start") as Double? ?: 0.0).dp
    val end = (padding?.get("end") as Double? ?: 0.0).dp
    val bottom = (padding?.get("bottom") as Double? ?: 0.0).dp

    val margin = attributes["margin"] as Map<*, *>?
    val mTop = (margin?.get("top") as Double? ?: 0.0).dp
    val mStart = (margin?.get("start") as Double? ?: 0.0).dp
    val mEnd = (margin?.get("end") as Double? ?: 0.0).dp
    val mBottom = (margin?.get("bottom") as Double? ?: 0.0).dp

    val color = Color((attributes["backgroundColor"] as String?)?.toColorInt() ?: Color.Unspecified.hashCode())
    return padding(start = mStart, top = mTop, end = mEnd, bottom = mBottom)
        .background(color)
        .padding(start = start, top = top, end = end, bottom = bottom)
        .modifyIf(fillMaxHeight) { fillMaxHeight() }
        .modifyIf(fillMaxWidth) { fillMaxWidth() }
        .modifyIf(height != null && !fillMaxHeight) { height(height!!) }
        .modifyIf(width != null && !fillMaxWidth) { width(width!!) }

}

fun Modifier.modifyIf(condition: Boolean, modify: Modifier.() -> Modifier) =
    if (condition) modify() else this
