package dev.openmobile.ui.components.inputs

import android.content.Intent
import android.net.Uri
import android.util.Log
import android.widget.Toast
import androidx.compose.foundation.BorderStroke
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.Button
import androidx.compose.material.ButtonDefaults
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.unit.dp
import androidx.core.graphics.toColorInt
import dev.openmobile.core.ManagerType
import dev.openmobile.core.OpenMobile
import dev.openmobile.ui.UIManager
import dev.openmobile.ui.utils.UIParser
import dev.openmobile.ui.utils.parseFromJson

@Composable
fun UIButton(json: Map<*, *>) {
    /* Todo: missing attributes to mapping
    enabled: Boolean = true,
    interactionSource: MutableInteractionSource = remember { MutableInteractionSource() },
    elevation: ButtonElevation? = ButtonDefaults.elevation(),
    shape: Shape = MaterialTheme.shapes.small,
    border: BorderStroke? = null,
    */
    val attributes = json["attributes"] as Map<*, *>
    val child = json["child"] as Map<*, *>
    val color = Color((attributes["color"] as String?)?.toColorInt() ?: Color.Black.hashCode())
    val rounded = attributes["rounded"] as Double? ?: 0.0
    val borderColor = Color((attributes["borderColor"] as String?)?.toColorInt() ?: Color.Transparent.hashCode())
    val borderWidth = attributes["borderWidth"] as Double? ?: 0.0
    val modifier = Modifier.parseFromJson(attributes["modifier"] as Map<*, *>?)
    val action = json["action"] as Map<*, *>?
    return Button(
        onClick = runAction(action),
        shape = RoundedCornerShape(rounded.toFloat()),
        colors = ButtonDefaults.buttonColors(backgroundColor = color),
        border = BorderStroke(borderWidth.dp, borderColor),
        modifier = modifier
    ) {
        UIParser(json = child)
    }
}

fun runAction(action: Map<*, *>?): () -> Unit = {
    action?.let {
        val type = action["type"] as String? ?: "url"
        val url = action["url"] as String? ?: ""
        val managerType = action["managerType"] as String? ?: ""
        val actionType = action["actionType"] as String? ?: ""
        val parameters = action["parameters"] as Map<*, *>?
        when(type) {
            "command" -> runCommand(managerType, actionType, parameters)
            "url",
            "deeplink" -> openUrl(url)
            "dismiss" -> dismiss()
        }
    }
}

fun runCommand(managerType: String, actionType: String, parameters: Map<*, *>?) {
    val manager = ManagerType.getByName(managerType)
    manager?.let { _ ->
        val action = manager.getAction(actionType)
        action?.let { _ ->
            OpenMobile.runAction(manager, action, parameters)
        }
    }
}

fun openUrl(value: String) {
    try {
        UIManager.client.context.startActivity(
            Intent(Intent.ACTION_VIEW).apply {
                data = Uri.parse(value)
                addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
            }
        )
    } catch (exception: Exception) {
        Toast.makeText(UIManager.client.context, exception.localizedMessage, Toast.LENGTH_LONG).show()
    }
}

fun dismiss() {
    Intent().also { intent ->
        intent.action = "dev.openmobile.ui.FINISH_ACTIVITY"
        UIManager.client.context.sendBroadcast(intent)
    }

}