package dev.netcode.blockchain;

import java.security.InvalidKeyException;
import java.security.KeyPair;
import java.security.SignatureException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import dev.netcode.blockchain.exceptions.InsufficientBalanceException;
import dev.netcode.security.encryption.RSAEncrypter;
import lombok.Getter;

public class Wallet {

    private KeyPair keyPair;
    @Getter private String walletID;
    private Map<String, TransactionOutput> UTXOs;

    public Wallet(){
        this.keyPair = RSAEncrypter.generateKeyPair(4096);
        this.walletID = RSAEncrypter.getFingerprint(this.keyPair.getPublic());
        this.UTXOs = new HashMap<String, TransactionOutput>();
    }

    public Transaction createTransaction(TransactionBlockChain blockchain, String recipient, double value) throws InsufficientBalanceException{
        if(getBalance() < value){
            throw new InsufficientBalanceException();
        }
        var inputs = new ArrayList<TransactionInput>();
        double total = 0;
        for(var entry : UTXOs.entrySet()){
            var output = entry.getValue();
            total += output.getValue();
            inputs.add(new TransactionInput(output.getID()));
            if(total >= value) break;
        }
        var transaction = new Transaction(this.keyPair.getPublic(), recipient, value, inputs);
        transaction.setSignature(sign(transaction.getData()));
        inputs.forEach(i -> UTXOs.remove(i.getTransactionOutputID()));
        return transaction;
    }

    public void updateBalance(TransactionBlockChain blockchain){
        this.UTXOs = blockchain.getUTXOs(walletID);
    }

    private String sign(String data) {
		try {
            return RSAEncrypter.sign(keyPair.getPrivate(),data);
        } catch (InvalidKeyException | SignatureException e) {
            e.printStackTrace();
        }
        return null;
	}

    public double getBalance() {
		return UTXOs.entrySet().stream().mapToDouble(i -> i.getValue().getValue()).sum();
	}

}
