package dev.lydtech.component.framework.debezium;

import java.io.InputStream;
import java.util.Optional;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.restassured.RestAssured;
import io.restassured.builder.RequestSpecBuilder;
import io.restassured.http.ContentType;
import io.restassured.specification.RequestSpecification;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public final class DebeziumAdminClient {

    private static String debeziumBaseUrl;
    private static final String CONNECTOR_PATH = "/connectors";
    private static RequestSpecification requestSpec;

    private static DebeziumAdminClient instance;

    private DebeziumAdminClient(){}

    public synchronized static DebeziumAdminClient getInstance() {
        if(instance==null) {
            String debeziumHost = Optional.ofNullable(System.getProperty("debezium.host"))
                    .orElse("localhost");
            String debeziumPort = Optional.ofNullable(System.getProperty("debezium.mapped.port"))
                    .orElseThrow(() -> new RuntimeException("debezium.mapped.port property not found"));
            debeziumBaseUrl = "http://" + debeziumHost + ":" + debeziumPort;
            log.info("Debezium base URL is: " + debeziumBaseUrl);
            requestSpec = new RequestSpecBuilder()
                    .setBaseUri(debeziumBaseUrl)
                    .build();
            instance = new DebeziumAdminClient();
        }
        return instance;
    }

    public String createConnector(String jsonFile) {
        try (InputStream in = Thread.currentThread().getContextClassLoader().getResourceAsStream(jsonFile)) {
            ObjectMapper mapper = new ObjectMapper();
            JsonNode jsonNode = mapper.readValue(in, JsonNode.class);
            String jsonString = mapper.writeValueAsString(jsonNode);
            log.debug("Creating Kafka Connect with config: "+jsonString);
            return postRequest(jsonString, CONNECTOR_PATH, 201);
        } catch (Exception e) {
            // Note this returns a 409 conflict if the connector already exists.
            log.error("Failed to create Kafka Connect connector: " + jsonFile, e);
            throw new RuntimeException(e);
        }
    }

    public void deleteConnector(String connectorName) {
        try {
            deleteRequest(CONNECTOR_PATH+"/"+connectorName, 204);
        } catch (Exception e) {
            // Note this returns a 409 conflict if rebalancing at time of delete.
            log.error("Failed to delete Kafka Connect connector: " + connectorName, e);
            throw new RuntimeException(e);
        }
    }

    private String postRequest(String request, String path, Integer expectedResponse) {
        String id = RestAssured.given()
                .spec(requestSpec)
                .body(request)
                .contentType(ContentType.JSON)
                .post(path)
                .then()
                .statusCode(expectedResponse)
                .extract()
                .response()
                .body()
                .path("id");
        return id;
    }

    private void deleteRequest(String path, Integer expectedResponse) {
        RestAssured.given()
                .spec(requestSpec)
                .delete(path)
                .then()
                .statusCode(expectedResponse);
    }
}
