package dev.lydtech.component.framework.extension;

import java.util.Arrays;
import java.util.List;
import java.util.Objects;

import com.github.dockerjava.api.DockerClient;
import com.github.dockerjava.api.command.ListContainersCmd;
import com.github.dockerjava.api.model.Container;
import com.github.dockerjava.api.model.ContainerPort;
import com.github.dockerjava.core.DefaultDockerClientConfig;
import com.github.dockerjava.core.DockerClientConfig;
import com.github.dockerjava.core.DockerClientImpl;
import com.github.dockerjava.httpclient5.ApacheDockerHttpClient;
import com.github.dockerjava.transport.DockerHttpClient;
import lombok.extern.slf4j.Slf4j;

import static dev.lydtech.component.framework.extension.TestContainersConfiguration.KAFKA_ENABLED;
import static java.util.Collections.singletonList;

@Slf4j
public final class DockerManager {

    protected static DockerClient getDockerClient() {
        log.info("Check if services are running");
        DockerClientConfig config = DefaultDockerClientConfig.createDefaultConfigBuilder()
                .withDockerHost("unix:///var/run/docker.sock")
                .build();
        DockerHttpClient httpClient = new ApacheDockerHttpClient.Builder()
                .dockerHost(config.getDockerHost())
                .sslConfig(config.getSSLConfig())
                .build();
        return DockerClientImpl.getInstance(config, httpClient);
    }

    /**
     * Skip TestContainers setup if:
     *
     * - Main container is running AND
     * - TestContainers container is not running
     */
    protected static boolean shouldPerformSetup(DockerClient dockerClient) {
        String mainContainerName = TestContainersConfiguration.CONTAINER_NAME_PREFIX + "-"+TestContainersConfiguration.SERVICE_NAME+"-0";
        List<Container> containers = dockerClient.listContainersCmd().exec();
        boolean mainContainerPresent = containers.stream().anyMatch(x -> Arrays.asList(x.getNames()).contains("/" + mainContainerName));
        boolean testContainersPresent = containers.stream().anyMatch(x -> Arrays.stream(x.getNames()).anyMatch(n -> n.startsWith("/testcontainers-ryuk")));

        log.info("Current container status - main({}): {}, tc: {}", mainContainerName, mainContainerPresent, testContainersPresent);

        return !(mainContainerPresent && !testContainersPresent);
    }

    protected static void captureDockerContainerPorts(DockerClient dockerClient) {
        String mainContainerName = TestContainersConfiguration.CONTAINER_NAME_PREFIX + "-"+TestContainersConfiguration.SERVICE_NAME+"-0";
        log.info("Service container name: "+mainContainerName);
        ListContainersCmd listContainersCmd = dockerClient.listContainersCmd();
        List<Container> serviceContainers = listContainersCmd.withNameFilter(singletonList(mainContainerName)).exec();
        if (serviceContainers.size() > 0) {
            Integer port = Arrays.stream(serviceContainers.get(0).getPorts())
                    .filter(x -> Objects.equals(x.getPrivatePort(), TestContainersConfiguration.SERVICE_PORT))
                    .map(ContainerPort::getPublicPort).findFirst()
                    .orElseThrow(() -> new RuntimeException("service port not found"));
            log.info("Service port is " + port);
            System.setProperty("service.port", port.toString());
        } else {
            throw new RuntimeException("Service container not found");
        }

        if(KAFKA_ENABLED) {
            String kafkaContainerName = TestContainersConfiguration.CONTAINER_NAME_PREFIX + "-kafka-0";
            List<Container> kafkaContainers = listContainersCmd.withNameFilter(singletonList(kafkaContainerName)).exec();
            if (kafkaContainers.size() == 1) {
                Integer port = Arrays.stream(kafkaContainers.get(0).getPorts())
                        .filter(x -> Objects.equals(x.getPrivatePort(), TestContainersConfiguration.KAFKA_PORT))
                        .map(ContainerPort::getPublicPort).findFirst()
                        .orElseThrow(() -> new RuntimeException("kafka port not found"));
                log.info("Kafka port is " + port);
                System.setProperty("kafka.port", port.toString());
            } else {
                throw new RuntimeException("Kafka container not found");
            }
        } else {
            log.info("Kafka broker is not enabled.");
        }
    }
}
