package dev.lukebemish.dynamicassetgenerator.api.client.generators;

import com.google.gson.JsonElement;
import com.mojang.blaze3d.platform.NativeImage;
import com.mojang.datafixers.util.Pair;
import com.mojang.serialization.Codec;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.DynamicOps;
import com.mojang.serialization.JsonOps;
import com.mojang.serialization.MapCodec;
import dev.lukebemish.dynamicassetgenerator.api.ResourceGenerationContext;
import dev.lukebemish.dynamicassetgenerator.api.ResourceGenerator;
import dev.lukebemish.dynamicassetgenerator.api.cache.CacheMetaCodec;
import dev.lukebemish.dynamicassetgenerator.api.cache.DataConsumer;
import dev.lukebemish.dynamicassetgenerator.impl.DynamicAssetGenerator;
import dev.lukebemish.dynamicassetgenerator.impl.ResourceCachingData;
import dev.lukebemish.dynamicassetgenerator.impl.client.ClientRegisters;
import dev.lukebemish.dynamicassetgenerator.impl.client.TexSourceCache;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.packs.resources.IoSupplier;
import org.jetbrains.annotations.ApiStatus;
import org.jspecify.annotations.NonNull;
import org.jspecify.annotations.Nullable;

import java.util.List;
import java.util.function.Function;

/**
 * Contains instructions for generating a single texture. Many implementations allow for nesting of further sources
 * within this; thus, to avoid the generation of duplicate sources, texture sources are cached, if possible, with their
 * key being their serialized form in JSON. If this information is not enough to uniquely identify the texture a source
 * will produce (for instance, if it uses information passed in a context), then a source should implement the caching
 * API as needed.
 */
public interface TexSource {
    @ApiStatus.Internal
    String METADATA_CACHE_KEY = "__dynamic_asset_generator_metadata";
    Codec<TexSource> CODEC = CacheMetaCodec.of(Codec.lazyInitialized(() -> new Codec<MapCodec<? extends TexSource>>() {
            @Override
            public <T> DataResult<Pair<MapCodec<? extends TexSource>, T>> decode(DynamicOps<T> ops, T input) {
                return ResourceLocation.CODEC.decode(ops, input).flatMap(keyValuePair -> !ClientRegisters.TEXSOURCES.containsKey(keyValuePair.getFirst())
                    ? DataResult.error(() -> "Unknown dynamic texture source type: " + keyValuePair.getFirst())
                    : DataResult.success(keyValuePair.mapFirst(ClientRegisters.TEXSOURCES::get)));
            }

            @Override
            public <T> DataResult<T> encode(MapCodec<? extends TexSource> input, DynamicOps<T> ops, T prefix) {
                ResourceLocation key = ClientRegisters.TEXSOURCES.inverse().get(input);
                if (key == null) {
                    return DataResult.error(() -> "Unregistered dynamic texture source type: " + input);
                }
                T toMerge = ops.createString(key.toString());
                return ops.mergeToPrimitive(prefix, toMerge);
            }
        }).dispatch(TexSource::codec, Function.identity()), List.of(
        CacheMetaCodec.SingleCacheType.of(new DataConsumer<>() {
            @Override
            @NonNull
            public <T1> DataResult<T1> encode(DynamicOps<T1> ops, TexSourceDataHolder data, TexSource object) {
                return object.cacheMetadata(ops, data);
            }
        }, METADATA_CACHE_KEY, TexSourceDataHolder.class),
        CacheMetaCodec.SingleCacheType.of(new DataConsumer<>() {
            @Override
            @NonNull
            public <T1> DataResult<T1> encode(DynamicOps<T1> ops, ResourceCachingData data, TexSource object) {
                return object.persistentCacheData(ops, data.context());
            }
        }, ResourceGenerator.PERSISTENT_CACHE_KEY, ResourceCachingData.class))
    );

    /**
     * Register a new type of texture source, alongside a codec to decode from JSON and encode to a cache key.
     * @param rl the identifier of this texture source type
     * @param codec can serialize and deserialize this texture source type
     * @param <T> the texture source type
     */
    static <T extends TexSource> void register(ResourceLocation rl, MapCodec<T> codec) {
        ClientRegisters.TEXSOURCES.put(rl, codec);
    }

    /**
     * If your texture source depends on runtime data context (anything stored in the {@link TexSourceDataHolder}), you will
     * need to override this method to return a unique key for the given context. This key will be used when caching
     * textures generated by this source. If your source should not be cached, return a DataResult.error.
     *
     * @param ops  DynamicOps to encode the unique key with.
     * @param data Data holder that the key is dependent on.
     * @return A success with a unique key for the given context, or an error if no key can be generated.
     */
    @NonNull
    @ApiStatus.Experimental
    default <T> DataResult<T> cacheMetadata(DynamicOps<T> ops, TexSourceDataHolder data) {
        return DataResult.success(ops.empty());
    }

    /**
     * If your texture source depends on loaded resources (anything stored in the {@link ResourceGenerationContext}),
     * you will need to override this method to return a unique key based on the actual, loaded resources. This key will
     * be used for caching <em>across reloads</em>, unlike that provided by {@link #cacheMetadata}. If your source
     * should not be cached, return a DataResult.error.
     *
     * @param ops DynamicOps to encode the unique key with.
     * @param context the context that the resource will be generated in. Resources can safely be accessed in this context
     * @return A success with a unique key for the given context, or an error if no key can be generated.
     */
    @NonNull
    @ApiStatus.Experimental
    default <T> DataResult<T> persistentCacheData(DynamicOps<T> ops, ResourceGenerationContext context) {
        return DataResult.success(ops.empty());
    }

    /**
     * @return a codec which can be used to serialize this source
     */
    @NonNull
    MapCodec<? extends TexSource> codec();

    /**
     * Provides a supplier for the texture this source will generate, or null if a texture cannot be provided. Should
     * be overridden, but not called; call {@link #getCachedSupplier} instead to support caching.
     * @param data context information passed by outer nesting texture sources; if you depend on this, you will want to
     *             implement the caching API (see {@link #cacheMetadata})
     * @param context context about the environment the texture is generating in
     * @return a supplier able to produce the texture, or null if the texture could not be produced.
     */
    @ApiStatus.OverrideOnly
    @Nullable
    IoSupplier<NativeImage> getSupplier(TexSourceDataHolder data, ResourceGenerationContext context);

    /**
     * Delegates to {@link #getSupplier}, but caches the result if possible. Should be used instead of the non-cached
     * version, but not extended.
     */
    @ApiStatus.NonExtendable
    default @Nullable IoSupplier<NativeImage> getCachedSupplier(TexSourceDataHolder data, ResourceGenerationContext context) {
        IoSupplier<NativeImage> wrapperImage = this.getSupplier(data, context);
        if (wrapperImage == null) return null;
        return () -> TexSourceCache.fromCache(wrapperImage, this, context, data);
    }

    /**
     * @return a string representation of this texture source, meant for logging and debugging purposes. Should
     * <em>not</em> be used for serialization or caching. Can be safely overridden to provide whatever information
     * is deemed useful.
     */
    default String stringify() {
        JsonElement element = CODEC.encodeStart(JsonOps.INSTANCE, this).result().orElse(null);
        if (element == null) return this.toString();
        return DynamicAssetGenerator.GSON_FLAT.toJson(element);
    }
}
