package dev.lukebemish.dynamicassetgenerator.api.client.generators.texsources.mask;

import com.mojang.serialization.Codec;
import com.mojang.serialization.MapCodec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import dev.lukebemish.dynamicassetgenerator.api.ResourceGenerationContext;
import dev.lukebemish.dynamicassetgenerator.api.client.generators.TexSource;
import dev.lukebemish.dynamicassetgenerator.api.client.generators.TexSourceDataHolder;
import dev.lukebemish.dynamicassetgenerator.impl.client.NativeImageHelper;
import org.jspecify.annotations.NonNull;
import org.jspecify.annotations.Nullable;

import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import net.minecraft.class_1011;
import net.minecraft.class_3542;
import net.minecraft.class_5253;
import net.minecraft.class_7367;

/**
 * A {@link TexSource} that detects edges in the alpha channel of a source.
 */
public final class EdgeMask implements TexSource {
    private static final int DEFAULT_CUTOFF = 128;
    private static final boolean DEFAULT_COUNT_OUTSIDE_FRAME = false;
    private static final List<Direction> DEFAULT_EDGES = Arrays.stream(Direction.values()).toList();

    public static final MapCodec<EdgeMask> CODEC = RecordCodecBuilder.mapCodec(i -> i.group(
            TexSource.CODEC.fieldOf("source").forGetter(EdgeMask::getSource),
            Codec.BOOL.optionalFieldOf("count_outside_frame", DEFAULT_COUNT_OUTSIDE_FRAME).forGetter(EdgeMask::isCountOutsideFrame),
            class_3542.method_28140(Direction::values).listOf().optionalFieldOf("edges", DEFAULT_EDGES).forGetter(EdgeMask::getEdges),
            Codec.INT.optionalFieldOf("cutoff", DEFAULT_CUTOFF).forGetter(EdgeMask::getCutoff)
    ).apply(i, EdgeMask::new));
    private final TexSource source;
    private final boolean countOutsideFrame;
    private final List<Direction> edges;
    private final int cutoff;

    private EdgeMask(TexSource source, boolean countOutsideFrame, List<Direction> edges, int cutoff) {
        this.source = source;
        this.countOutsideFrame = countOutsideFrame;
        this.edges = edges;
        this.cutoff = cutoff;
    }

    @Override
    public @NonNull MapCodec<? extends TexSource> codec() {
        return CODEC;
    }

    @Override
    public @Nullable class_7367<class_1011> getSupplier(TexSourceDataHolder data, ResourceGenerationContext context) {
        class_7367<class_1011> input = this.source.getCachedSupplier(data, context);
        if (input == null) {
            data.getLogger().error("Texture given was nonexistent...\n{}", this.source.stringify());
            return null;
        }
        int[] xs = edges.stream().mapToInt(e -> e.x).toArray();
        int[] ys = edges.stream().mapToInt(e -> e.y).toArray();
        return () -> {
            try (class_1011 inImg = input.get()) {
                int width = inImg.method_4307();
                int height = inImg.method_4323();
                class_1011 out = NativeImageHelper.of(class_1011.class_1012.field_4997, width, height, false);
                for (int x = 0; x < width; x++) {
                    for (int y = 0; y < height; y++) {
                        boolean isEdge = false;
                        int color = inImg.method_4315(x, y);
                        if (class_5253.class_8045.method_48342(color) >= cutoff) {
                            for (int i = 0; i < xs.length; i++) {
                                int x1 = xs[i] + x;
                                int y1 = ys[i] + y;
                                if ((countOutsideFrame && (x1 < 0 || y1 < 0 || x1 > width - 1 || y1 > width - 1)) ||
                                        class_5253.class_8045.method_48342(inImg.method_4315(x1, y1)) < cutoff)
                                    isEdge = true;
                            }
                        }

                        if (isEdge)
                            out.method_4305(x, y, 0xFFFFFFFF);
                        else
                            out.method_4305(x, y, 0);
                    }
                }
                return out;
            }
        };
    }

    public TexSource getSource() {
        return source;
    }

    public boolean isCountOutsideFrame() {
        return countOutsideFrame;
    }

    public List<Direction> getEdges() {
        return edges;
    }

    public int getCutoff() {
        return cutoff;
    }

    /**
     * Represents the direction of a neighboring pixel in 2D space.
     */
    public enum Direction implements class_3542 {
        NORTH(0, -1),
        NORTHEAST(1, -1),
        EAST(1, 0),
        SOUTHEAST(1, 1),
        SOUTH(0, 1),
        SOUTHWEST(-1, 1),
        WEST(-1, 0),
        NORTHWEST(-1, -1);

        public final int x;
        public final int y;

        Direction(int x, int y) {
            this.x = x;
            this.y = y;
        }

        @Override
        public @NonNull String method_15434() {
            return name().toLowerCase(Locale.ROOT);
        }
    }

    public static class Builder {
        private TexSource source;
        private boolean countOutsideFrame = DEFAULT_COUNT_OUTSIDE_FRAME;
        private List<Direction> edges = DEFAULT_EDGES;
        private int cutoff = DEFAULT_CUTOFF;

        /**
         * Sets the input texture.
         */
        public Builder setSource(TexSource source) {
            this.source = source;
            return this;
        }

        /**
         * Sets whether to count pixels outside the frame as opaque. Defaults to false.
         */
        public Builder setCountOutsideFrame(boolean countOutsideFrame) {
            this.countOutsideFrame = countOutsideFrame;
            return this;
        }

        /**
         * Sets the directions to look, relative to opaque pixels, for edges. Defaults to all directions.
         */
        public Builder setEdges(List<Direction> edges) {
            this.edges = edges;
            return this;
        }

        /**
         * Sets the cutoff for what is considered opaque. Defaults to 128.
         */
        public Builder setCutoff(int cutoff) {
            this.cutoff = cutoff;
            return this;
        }

        public EdgeMask build() {
            Objects.requireNonNull(source);
            return new EdgeMask(source, countOutsideFrame, edges, cutoff);
        }
    }
}
