package dev.lukebemish.dynamicassetgenerator.api.client.generators.texsources.mask;

import com.mojang.serialization.MapCodec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import dev.lukebemish.dynamicassetgenerator.api.ResourceGenerationContext;
import dev.lukebemish.dynamicassetgenerator.api.client.generators.TexSource;
import dev.lukebemish.dynamicassetgenerator.api.client.generators.TexSourceDataHolder;
import dev.lukebemish.dynamicassetgenerator.api.client.image.ImageUtils;
import dev.lukebemish.dynamicassetgenerator.api.colors.Channel;
import dev.lukebemish.dynamicassetgenerator.api.colors.operations.PointwiseOperation;
import org.jspecify.annotations.NonNull;
import org.jspecify.annotations.Nullable;

import java.util.List;
import java.util.Objects;
import net.minecraft.class_1011;
import net.minecraft.class_7367;

/**
 * A {@link TexSource} that extracts a single channel from a texture.
 */
@SuppressWarnings("unused")
public final class ChannelMask implements TexSource {
    public static final MapCodec<ChannelMask> CODEC = RecordCodecBuilder.mapCodec(i -> i.group(
            TexSource.CODEC.fieldOf("source").forGetter(ChannelMask::getSource),
            Channel.CODEC.fieldOf("channel").forGetter(ChannelMask::getChannel)
    ).apply(i, ChannelMask::new));
    private final TexSource source;
    private final Channel channel;

    private ChannelMask(TexSource source, Channel channel) {
        this.source = source;
        this.channel = channel;
    }

    @Override
    public @NonNull MapCodec<? extends TexSource> codec() {
        return CODEC;
    }

    @Override
    public @Nullable class_7367<class_1011> getSupplier(TexSourceDataHolder data, ResourceGenerationContext context) {
        class_7367<class_1011> input = this.source.getCachedSupplier(data, context);
        if (input == null) {
            data.getLogger().error("Texture given was nonexistent...\n{}", this.source.stringify());
            return null;
        }
        return () -> {
            PointwiseOperation.Unary<Integer> operation = PointwiseOperation.Unary.chain(
                    channel.makeOperation(),
                    (c, i) -> ((c & 0xFF) << 24) | 0xFFFFFF
            );
            try (class_1011 inImg = input.get()) {
                return ImageUtils.generateScaledImage(operation, List.of(inImg));
            }
        };
    }

    public TexSource getSource() {
        return source;
    }

    public Channel getChannel() {
        return channel;
    }

    public static class Builder {
        private TexSource source;
        private Channel channel;

        /**
         * Sets the input texture.
         */
        public Builder setSource(TexSource source) {
            this.source = source;
            return this;
        }

        /**
         * Sets the channel to extract.
         */
        public Builder setChannel(Channel channel) {
            this.channel = channel;
            return this;
        }

        public ChannelMask build() {
            Objects.requireNonNull(source);
            Objects.requireNonNull(channel);
            return new ChannelMask(source, channel);
        }
    }
}
