package dev.lukebemish.dynamicassetgenerator.api.client.generators;

import com.mojang.serialization.DataResult;
import com.mojang.serialization.DynamicOps;
import com.mojang.serialization.MapCodec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import dev.lukebemish.dynamicassetgenerator.api.ResourceGenerationContext;
import dev.lukebemish.dynamicassetgenerator.api.ResourceGenerator;
import dev.lukebemish.dynamicassetgenerator.impl.DynamicAssetGenerator;
import org.jspecify.annotations.NonNull;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Set;
import net.minecraft.class_1011;
import net.minecraft.class_2960;
import net.minecraft.class_7367;

/**
 * A resource generator that generates a PNG texture as specified by a {@link TexSource}. As {@link TexSource}s are
 * cached in memory to avoid regenerating duplicate parts of a texture, this generator should only be used with a
 * {@link dev.lukebemish.dynamicassetgenerator.api.client.AssetResourceCache}.
 */
public class TextureGenerator implements ResourceGenerator {
    public static final MapCodec<TextureGenerator> CODEC = RecordCodecBuilder.mapCodec(instance -> instance.group(
            class_2960.field_25139.fieldOf("output_location").forGetter(dyn->dyn.outputLocation),
            TexSource.CODEC.fieldOf("input").forGetter(dyn->dyn.input)
    ).apply(instance, TextureGenerator::new));

    private final class_2960 outputLocation;
    private final TexSource input;

    /**
     * @param outputLocation the location to generate a texture at, excluding the "textures/" prefix or ".png" extension
     * @param source the texture source to generate
     */
    public TextureGenerator(@NonNull class_2960 outputLocation, @NonNull TexSource source) {
        this.input = source;
        this.outputLocation = outputLocation;
    }

    @Override
    public class_7367<InputStream> get(class_2960 outRl, ResourceGenerationContext context) {
        class_7367<class_1011> imageGetter = this.input.getCachedSupplier(new TexSourceDataHolder(), context);
        if (imageGetter == null) return null;
        return () -> {
            try (class_1011 image = imageGetter.get()) {
                return new ByteArrayInputStream(image.method_24036());
            } catch (IOException e) {
                DynamicAssetGenerator.LOGGER.error("Could not write image to stream for source {}: {}", input.stringify(), outRl, e);
                throw e;
            } catch (Exception remainder) {
                DynamicAssetGenerator.LOGGER.error("Unknown issue creating texture for output {} with source {}", outRl, input.stringify(), remainder);
                throw new IOException(remainder);
            }
        };
    }

    @Override
    public @NonNull <T> DataResult<T> persistentCacheData(DynamicOps<T> ops, class_2960 location, ResourceGenerationContext context) {
        return DataResult.success(ops.empty()); // The actual persistent data is recorded while encoding the held texture source
    }

    @Override
    public @NonNull Set<class_2960> getLocations(ResourceGenerationContext context) {
        return Set.of(getOutputLocation());
    }

    private class_2960 getOutputLocation() {
        return class_2960.method_60655(this.outputLocation.method_12836(), "textures/"+this.outputLocation.method_12832()+".png");
    }

    @Override
    public MapCodec<? extends ResourceGenerator> codec() {
        return CODEC;
    }
}
