package dev.lukebemish.dynamicassetgenerator.api.client;

import com.google.gson.JsonObject;
import com.mojang.datafixers.util.Pair;
import com.mojang.serialization.MapCodec;
import dev.lukebemish.dynamicassetgenerator.api.ResourceGenerationContext;
import dev.lukebemish.dynamicassetgenerator.api.TrackingResourceSource;
import dev.lukebemish.dynamicassetgenerator.api.cache.CacheMetaJsonOps;
import dev.lukebemish.dynamicassetgenerator.api.client.generators.TexSource;
import dev.lukebemish.dynamicassetgenerator.api.client.generators.TexSourceDataHolder;
import dev.lukebemish.dynamicassetgenerator.api.client.generators.TextureMetaGenerator;
import dev.lukebemish.dynamicassetgenerator.impl.DynamicAssetGenerator;
import dev.lukebemish.dynamicassetgenerator.impl.ResourceCachingData;
import dev.lukebemish.dynamicassetgenerator.impl.client.ForegroundExtractor;
import dev.lukebemish.dynamicassetgenerator.impl.client.TexSourceCache;
import dev.lukebemish.dynamicassetgenerator.impl.client.platform.ClientServices;
import dev.lukebemish.dynamicassetgenerator.impl.mixin.SpriteSourcesAccessor;
import dev.lukebemish.dynamicassetgenerator.impl.util.ResourceUtils;
import org.jetbrains.annotations.ApiStatus;
import org.jspecify.annotations.NonNull;
import org.jspecify.annotations.Nullable;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.function.Supplier;
import net.minecraft.class_1011;
import net.minecraft.class_1079;
import net.minecraft.class_2960;
import net.minecraft.class_3264;
import net.minecraft.class_3300;
import net.minecraft.class_3518;
import net.minecraft.class_7367;
import net.minecraft.class_7368;
import net.minecraft.class_7764;
import net.minecraft.class_7771;
import net.minecraft.class_7948;
import net.minecraft.class_7951;

/**
 * A sprite source which makes use of {@link TexSource}s to provide sprites at resource pack load. May be more reliable
 * than a {@link AssetResourceCache} for generating sprites based off of textures added by other mods which use runtime
 * resource generation techniques.
 */
public interface SpriteProvider<T extends SpriteProvider<T>> {
    /**
     * @return a map of texture location, not including the {@code "textures/"} prefix or file extension, to texture source
     */
    Map<class_2960, TexSource> getSources(ResourceGenerationContext context);

    /**
     * @return a unique identifier for this sprite source type
     */
    class_2960 getLocation();

    /**
     * Registers a sprite source type.
     * @param location the location which this sprite source type can be referenced from a texture atlas JSON file with
     * @param codec a codec to provide instances of the type
     */
    static <T extends SpriteProvider<T>> void register(class_2960 location, MapCodec<T> codec) {
        ClientServices.PLATFORM_CLIENT.addSpriteSource(location, codec.xmap(SpriteProvider::wrap, Wrapper::unwrap));
    }

    /**
     * Registers a sprite source type
     * @param location the location which this sprite source type can be referenced from a texture atlas JSON file with
     * @param constructor supplies instances of this sprite source type
     */
    static <T extends SpriteProvider<T>> void register(class_2960 location, Supplier<T> constructor) {
        ClientServices.PLATFORM_CLIENT.addSpriteSource(location, MapCodec.unit(() -> constructor.get().wrap()));
    }

    /**
     * Will be run before generation starts. Allows for clearing of anything that saves state (caches or the like).
     * Implementations should call the super method to clear texture source and palette transfer caches.
     * @param context context for the generation that will occur after this source is reset
     */
    default void reset(ResourceGenerationContext context) {
        TexSourceCache.reset(context);
        ForegroundExtractor.reset(context);
    }

    default void run(class_3300 resourceManager, class_7948.class_7949 output, class_2960 cacheName) {
        ResourceGenerationContext context = new ResourceGenerationContext() {
            private final ResourceSource source = ResourceGenerationContext.ResourceSource.filtered(pack -> true, class_3264.field_14188, resourceManager::method_29213)
                .fallback(new ResourceSource() {
                    @Override
                    public @Nullable class_7367<InputStream> getResource(@NonNull class_2960 location) {
                        return resourceManager.method_14486(location).<class_7367<InputStream>>map(r -> r::method_14482).orElse(null);
                    }

                    @Override
                    public List<class_7367<InputStream>> getResourceStack(@NonNull class_2960 location) {
                        return resourceManager.method_14489(location).stream().<class_7367<InputStream>>map(r -> r::method_14482).toList();
                    }

                    @Override
                    public Map<class_2960, class_7367<InputStream>> listResources(@NonNull String path, @NonNull Predicate<class_2960> filter) {
                        Map<class_2960, class_7367<InputStream>> map = new HashMap<>();
                        resourceManager.method_14488(path, filter).forEach((rl, r) -> map.put(rl, r::method_14482));
                        return map;
                    }

                    @Override
                    public Map<class_2960, List<class_7367<InputStream>>> listResourceStacks(@NonNull String path, @NonNull Predicate<class_2960> filter) {
                        Map<class_2960, List<class_7367<InputStream>>> map = new HashMap<>();
                        resourceManager.method_41265(path, filter).forEach((rl, r) -> map.put(rl, r.stream().<class_7367<InputStream>>map(i -> i::method_14482).toList()));
                        return map;
                    }

                    @Override
                    public @NonNull Set<String> getNamespaces() {
                        return resourceManager.method_14487();
                    }
                });

            @Override
            public @NonNull class_2960 getCacheName() {
                return cacheName;
            }

            @Override
            public ResourceSource getResourceSource() {
                return source;
            }
        };

        this.reset(context);

        Map<class_2960, TexSource> sources = getSources(context);

        sources.forEach((rl, texSource) -> {
            var trackingSource = TrackingResourceSource.of(context.getResourceSource(), "textures", ".png");
            ResourceGenerationContext trackingContext = context.withResourceSource(trackingSource);
            var dataHolder = new TexSourceDataHolder();
            class_7367<class_1011> imageSupplier = ResourceUtils.wrapSafeData(
                class_2960.method_60655(rl.method_12836(), "textures/"+rl.method_12832()+".png"),
                (r, c) -> texSource.getCachedSupplier(dataHolder, c),
                trackingContext,
                im -> {
                    try (var image = im) {
                        return new ByteArrayInputStream(image.method_24036());
                    }
                },
                is -> {
                    try (var input = is) {
                        return class_1011.method_4309(input);
                    }
                },
                (r, c) -> {
                    CacheMetaJsonOps ops = new CacheMetaJsonOps();
                    ops.putData(ResourceCachingData.class, new ResourceCachingData(r, c));
                    return TexSource.CODEC.encodeStart(ops, texSource).result().map(DynamicAssetGenerator.GSON_FLAT::toJson).orElse(null);
                }
            );
            output.method_47670(rl, spriteResourceLoader -> {
                try {
                    if (imageSupplier == null) {
                        throw new IOException("No image supplier");
                    }
                    final class_1011 image = imageSupplier.get();
                    class_1079 section = class_1079.field_21768;
                    if (!trackingSource.getTouchedTextures().isEmpty()) {
                        TextureMetaGenerator.AnimationGenerator generator = new TextureMetaGenerator.AnimationGenerator.Builder().build();
                        List<Pair<class_2960, JsonObject>> animations = new ArrayList<>();
                        for (class_2960 touchedTexture : trackingSource.getTouchedTextures()) {
                            var resource = context.getResourceSource().getResource(class_2960.method_60655(touchedTexture.method_12836(), "textures/"+touchedTexture.method_12832()+".png.mcmeta"));
                            if (resource == null) {
                                animations.add(new Pair<>(touchedTexture, null));
                                continue;
                            }
                            try (var reader = new BufferedReader(new InputStreamReader(resource.get()))) {
                                JsonObject json = DynamicAssetGenerator.GSON.fromJson(reader, JsonObject.class);
                                JsonObject animation = class_3518.method_15296(json, class_1079.field_32974);
                                animations.add(new Pair<>(touchedTexture, animation));
                            } catch (Exception ignored) {
                                animations.add(new Pair<>(touchedTexture, null));
                            }
                        }
                        JsonObject built = generator.apply(animations);
                        if (built != null) {
                            try {
                                section = class_1079.field_5337.method_4692(built);
                            } catch (Exception ignored) {
                            }
                        }
                    }
                    class_7771 frameSize = new class_7771(image.method_4307(), image.method_4323());
                    if (section != class_1079.field_21768) {
                        frameSize = section.method_24143(image.method_4307(), image.method_4323());
                    }
                    return new class_7764(rl, frameSize, image, new class_7368.class_8622().method_52448(class_1079.field_5337, section).method_52447());
                } catch (IOException e) {
                    DynamicAssetGenerator.LOGGER.error("Failed to generate texture for sprite source type "+getLocation()+" at "+rl+": ", e);
                    return null;
                }
            });
        });
    }

    @SuppressWarnings("unchecked")
    default Wrapper<T> wrap() {
        return new Wrapper<>((T) this, null);
    }

    final class Wrapper<T extends SpriteProvider<T>> implements class_7948 {
        private final T source;
        private final @Nullable class_2960 location;

        private Wrapper(T source, @Nullable class_2960 location) {
            this.source = source;
            this.location = location;
        }

        @Override
        public void method_47673(class_3300 resourceManager, class_7949 output) {
            class_2960 cacheName = this.location == null ?
                source.getLocation() :
                source.getLocation().method_48331("__"+this.location.method_12836()+"__"+this.location.method_12832());
            source.run(resourceManager, output, cacheName);
        }

        @Override
        public @NonNull class_7951 method_47672() {
            return SpriteSourcesAccessor.dynamic_asset_generator$getTypes().get(source.getLocation());
        }

        public T unwrap() {
            return source;
        }

        @ApiStatus.Internal
        public Wrapper<T> withLocation(class_2960 atlasLocation) {
            return new Wrapper<>(source, atlasLocation);
        }
    }
}
