package dev.lukebemish.dynamicassetgenerator.api;

import org.jetbrains.annotations.Contract;
import org.jspecify.annotations.NonNull;
import org.jspecify.annotations.Nullable;

import java.io.InputStream;
import java.util.*;
import java.util.function.Predicate;
import net.minecraft.class_2960;
import net.minecraft.class_7367;

/**
 * A resource source that tracks the resources that have been accessed.
 */
public class TrackingResourceSource implements ResourceGenerationContext.ResourceSource {

    private final ResourceGenerationContext.ResourceSource delegate;
    private final List<class_2960> touchedTextures = new ArrayList<>();
    private final Set<class_2960> touchedTexturesSet = new HashSet<>();
    private final List<class_2960> view = Collections.unmodifiableList(touchedTextures);

    private final String prefix;
    private final String suffix;

    /**
     * @return the list of resources that have been touched by generators seeing this
     */
    public List<class_2960> getTouchedTextures() {
        return view;
    }

    private TrackingResourceSource(ResourceGenerationContext.ResourceSource delegate, String prefix, String suffix) {
        this.delegate = delegate;
        this.prefix = prefix.endsWith("/")? prefix : prefix+"/";
        this.suffix = suffix;
    }

    /**
     * Creates a new tracking resource source.
     * @param delegate the delegate resource source
     * @param prefix the prefix to track resources within, such as {@code "textures}
     * @param suffix the suffix to track resources ending with, such as {@code ".png"}
     */
    @Contract("_, _, _ -> new")
    public static TrackingResourceSource of(ResourceGenerationContext.ResourceSource delegate, String prefix, String suffix) {
        return new TrackingResourceSource(delegate, prefix, suffix);
    }

    private void addLocation(class_2960 location) {
        if (location.method_12832().startsWith(prefix) && location.method_12832().endsWith(suffix)) {
            unsafeAddLocation(new class_2960(location.method_12836(), location.method_12832().substring(prefix.length(), location.method_12832().length() - suffix.length())));
        }
    }

    private synchronized void unsafeAddLocation(class_2960 location) {
        if (touchedTexturesSet.add(location)) {
            touchedTextures.add(location);
        }
    }

    @Override
    public @Nullable class_7367<InputStream> getResource(@NonNull class_2960 location) {
        addLocation(location);
        return delegate.getResource(location);
    }

    @Override
    public List<class_7367<InputStream>> getResourceStack(@NonNull class_2960 location) {
        addLocation(location);
        return delegate.getResourceStack(location);
    }

    @Override
    public Map<class_2960, class_7367<InputStream>> listResources(@NonNull String path, @NonNull Predicate<class_2960> filter) {
        var resources = delegate.listResources(path, filter);
        resources.keySet().forEach(this::addLocation);
        return resources;
    }

    @Override
    public Map<class_2960, List<class_7367<InputStream>>> listResourceStacks(@NonNull String path, @NonNull Predicate<class_2960> filter) {
        var resources = delegate.listResourceStacks(path, filter);
        resources.keySet().forEach(this::addLocation);
        return resources;
    }

    @Override
    public @NonNull Set<String> getNamespaces() {
        return delegate.getNamespaces();
    }
}
