package dev.lukebemish.dynamicassetgenerator.api.client.generators.texsources;

import com.mojang.serialization.Codec;
import com.mojang.serialization.MapCodec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import dev.lukebemish.dynamicassetgenerator.api.ResourceGenerationContext;
import dev.lukebemish.dynamicassetgenerator.api.client.generators.TexSource;
import dev.lukebemish.dynamicassetgenerator.api.client.generators.TexSourceDataHolder;
import dev.lukebemish.dynamicassetgenerator.api.client.image.ImageUtils;
import dev.lukebemish.dynamicassetgenerator.impl.client.NativeImageHelper;
import org.jspecify.annotations.NonNull;
import org.jspecify.annotations.Nullable;

import java.util.Objects;
import net.minecraft.class_1011;
import net.minecraft.class_7367;

/**
 * A {@link TexSource} that transforms another {@link TexSource} by rotating and/or flipping it.
 */
public final class TransformSource implements TexSource {
    private static final int DEFAULT_ROTATE = 0;
    private static final boolean DEFAULT_FLIP = false;

    public static final MapCodec<TransformSource> CODEC = RecordCodecBuilder.mapCodec(instance -> instance.group(
            TexSource.CODEC.fieldOf("input").forGetter(TransformSource::getInput),
            Codec.INT.optionalFieldOf("rotate", DEFAULT_ROTATE).forGetter(TransformSource::getRotate),
            Codec.BOOL.optionalFieldOf("flip", DEFAULT_FLIP).forGetter(TransformSource::isFlip)
    ).apply(instance, TransformSource::new));
    private final TexSource input;
    private final int rotate;
    private final boolean flip;

    private TransformSource(TexSource input, int rotate, boolean flip) {
        this.input = input;
        this.rotate = rotate;
        this.flip = flip;
    }

    @Override
    public @NonNull MapCodec<? extends TexSource> codec() {
        return CODEC;
    }

    @Override
    public @Nullable class_7367<class_1011> getSupplier(TexSourceDataHolder data, ResourceGenerationContext context) {
        class_7367<class_1011> input = this.getInput().getCachedSupplier(data, context);
        if (input == null) {
            data.getLogger().error("Texture given was nonexistent...\n{}", this.getInput());
            return null;
        }
        return () -> {
            class_1011 output = input.get();
            for (int i = 0; i < this.getRotate(); i++) {
                output = clockwiseRotate(output);
            }
            if (this.isFlip()) {
                class_1011 output2 = NativeImageHelper.of(output.method_4318(), output.method_4307(), output.method_4323(), false);
                for (int x = 0; x < output.method_4307(); x++) {
                    for (int y = 0; y < output.method_4323(); y++) {
                        output2.method_4305((output.method_4307() - 1 - x), y, ImageUtils.safeGetPixelABGR(output, x, y));
                    }
                }
                output.close();
                output = output2;
            }
            return output;
        };
    }

    private static class_1011 clockwiseRotate(class_1011 input) {
        int w = input.method_4307();
        int h = input.method_4323();
        class_1011 output = NativeImageHelper.of(input.method_4318(), h, w, false);
        for (int y = 0; y < h; y++)
            for (int x = 0; x < w; x++)
                //noinspection SuspiciousNameCombination
                output.method_4305(y, w - x - 1, ImageUtils.safeGetPixelABGR(input, x, y));
        input.close();
        return output;
    }

    public TexSource getInput() {
        return input;
    }

    public int getRotate() {
        return rotate;
    }

    public boolean isFlip() {
        return flip;
    }

    public static class Builder {
        private TexSource input;
        private int rotate = DEFAULT_ROTATE;
        private boolean flip = DEFAULT_FLIP;

        /**
         * Provides the input {@link TexSource} to transform.
         */
        public Builder setInput(TexSource input) {
            this.input = input;
            return this;
        }

        /**
         * Sets the number of 90 degree clockwise rotations to apply to the input. Defaults to 0
         */
        public Builder setRotate(int rotate) {
            this.rotate = rotate;
            return this;
        }

        /**
         * Sets whether to flip the input horizontally after rotations have been applied. Defaults to false
         */
        public Builder setFlip(boolean flip) {
            this.flip = flip;
            return this;
        }

        public TransformSource build() {
            Objects.requireNonNull(input);
            return new TransformSource(input, rotate, flip);
        }
    }
}
