package dev.langchain4j.store.embedding;

import dev.langchain4j.store.embedding.milvus.MilvusCollectionDescription;
import io.milvus.client.MilvusServiceClient;
import io.milvus.grpc.FlushResponse;
import io.milvus.grpc.MutationResult;
import io.milvus.grpc.QueryResults;
import io.milvus.grpc.SearchResults;
import io.milvus.param.R;
import io.milvus.param.RpcStatus;
import io.milvus.param.collection.FlushParam;
import io.milvus.param.collection.LoadCollectionParam;
import io.milvus.param.dml.InsertParam;
import io.milvus.param.dml.QueryParam;
import io.milvus.param.dml.SearchParam;
import io.milvus.response.QueryResultsWrapper;
import io.milvus.response.SearchResultsWrapper;

import java.util.List;

import static dev.langchain4j.store.embedding.CollectionRequestBuilder.*;

class CollectionOperationsExecutor {

    public static void flush(MilvusServiceClient milvusClient, String collectionName) {
        FlushParam request = buildFlushRequest(collectionName);
        R<FlushResponse> response = milvusClient.flush(request);
        checkResponseNotFailed(response);
    }

    public static void insert(MilvusServiceClient milvusClient, List<InsertParam.Field> fields, String collectionName) {
        InsertParam request = buildInsertRequest(fields, collectionName);
        R<MutationResult> response = milvusClient.insert(request);
        checkResponseNotFailed(response);
    }

    public static void loadCollectionInMemory(MilvusServiceClient milvusClient, String collectionName) {
        LoadCollectionParam request = buildLoadCollectionInMemoryRequest(collectionName);
        R<RpcStatus> response = milvusClient.loadCollection(request);
        checkResponseNotFailed(response);
    }

    public static SearchResultsWrapper search(MilvusServiceClient milvusClient, SearchParam searchRequest) {
        R<SearchResults> response = milvusClient.search(searchRequest);
        checkResponseNotFailed(response);

        return new SearchResultsWrapper(response.getData().getResults());
    }

    public static QueryResultsWrapper queryForVectors(MilvusServiceClient milvusClient,
                                                     MilvusCollectionDescription collectionDescription,
                                                     List<String> rowIds,
                                                     String consistencyLevel) {
        QueryParam request = buildQueryRequest(rowIds, collectionDescription, consistencyLevel);
        R<QueryResults> response = milvusClient.query(request);
        checkResponseNotFailed(response);

        return new QueryResultsWrapper(response.getData());
    }

    private static <T> void checkResponseNotFailed(R<T> response) {
        if (response == null) {
            throw new RequestToMilvusFailedException("Request to Milvus DB failed. Response is null");
        } else if (response.getStatus() != R.Status.Success.getCode()) {
            String message = String.format("Request to Milvus DB failed. Response status:'%d'.%n", response.getStatus());
            throw new RequestToMilvusFailedException(message, response.getException());
        }
    }


}
