/*
 * Decompiled with CFR 0.152.
 */
package net.luminis.quic.tls;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import net.luminis.quic.Bytes;
import net.luminis.quic.InvalidIntegerEncodingException;
import net.luminis.quic.ProtocolError;
import net.luminis.quic.QuicConstants;
import net.luminis.quic.Role;
import net.luminis.quic.TransportParameters;
import net.luminis.quic.VariableLengthInteger;
import net.luminis.quic.Version;
import net.luminis.quic.log.Logger;
import net.luminis.tls.alert.DecodeErrorException;
import net.luminis.tls.extension.Extension;
import net.luminis.tls.util.ByteUtils;

public class QuicTransportParametersExtension
extends Extension {
    private static final int MINIMUM_EXTENSION_LENGTH = 2;
    public static final int CODEPOINT_IETFDRAFT = 65445;
    public static final int CODEPOINT_V1 = 57;
    private final Version quicVersion;
    private Role senderRole;
    private byte[] data;
    private TransportParameters params;
    private Integer discardTransportParameterSize;

    public static boolean isCodepoint(Version quicVersion, int extensionType) {
        if (quicVersion == Version.QUIC_version_1) {
            return extensionType == 57;
        }
        return extensionType == 65445;
    }

    public QuicTransportParametersExtension() {
        this(Version.getDefault());
    }

    public QuicTransportParametersExtension(Version quicVersion) {
        this.quicVersion = quicVersion;
        this.params = new TransportParameters();
    }

    public QuicTransportParametersExtension(Version quicVersion, TransportParameters params, Role senderRole) {
        this.quicVersion = quicVersion;
        this.params = params;
        this.senderRole = senderRole;
    }

    @Override
    public byte[] getBytes() {
        if (this.data == null) {
            this.serialize();
        }
        return this.data;
    }

    public void addDiscardTransportParameter(int parameterSize) {
        this.discardTransportParameterSize = parameterSize;
    }

    private void serialize() {
        ByteBuffer buffer = ByteBuffer.allocate(100 + (this.discardTransportParameterSize != null ? this.discardTransportParameterSize : 0));
        buffer.putShort((short)(this.quicVersion == Version.QUIC_version_1 ? 57 : 65445));
        buffer.putShort((short)0);
        if (this.senderRole == Role.Server) {
            this.addTransportParameter(buffer, QuicConstants.TransportParameterId.original_destination_connection_id, this.params.getOriginalDestinationConnectionId());
        }
        this.addTransportParameter(buffer, QuicConstants.TransportParameterId.max_idle_timeout, this.params.getMaxIdleTimeout());
        if (this.senderRole == Role.Server && this.params.getStatelessResetToken() != null) {
            this.addTransportParameter(buffer, QuicConstants.TransportParameterId.stateless_reset_token, this.params.getStatelessResetToken());
        }
        this.addTransportParameter(buffer, QuicConstants.TransportParameterId.max_udp_payload_size, (long)this.params.getMaxUdpPayloadSize());
        this.addTransportParameter(buffer, QuicConstants.TransportParameterId.initial_max_data, this.params.getInitialMaxData());
        this.addTransportParameter(buffer, QuicConstants.TransportParameterId.initial_max_stream_data_bidi_local, this.params.getInitialMaxStreamDataBidiLocal());
        this.addTransportParameter(buffer, QuicConstants.TransportParameterId.initial_max_stream_data_bidi_remote, this.params.getInitialMaxStreamDataBidiRemote());
        this.addTransportParameter(buffer, QuicConstants.TransportParameterId.initial_max_stream_data_uni, this.params.getInitialMaxStreamDataUni());
        this.addTransportParameter(buffer, QuicConstants.TransportParameterId.initial_max_streams_bidi, this.params.getInitialMaxStreamsBidi());
        this.addTransportParameter(buffer, QuicConstants.TransportParameterId.initial_max_streams_uni, this.params.getInitialMaxStreamsUni());
        this.addTransportParameter(buffer, QuicConstants.TransportParameterId.ack_delay_exponent, (long)this.params.getAckDelayExponent());
        this.addTransportParameter(buffer, QuicConstants.TransportParameterId.max_ack_delay, (long)this.params.getMaxAckDelay());
        this.addTransportParameter(buffer, QuicConstants.TransportParameterId.active_connection_id_limit, (long)this.params.getActiveConnectionIdLimit());
        this.addTransportParameter(buffer, QuicConstants.TransportParameterId.initial_source_connection_id, this.params.getInitialSourceConnectionId());
        if (this.senderRole == Role.Server && this.params.getRetrySourceConnectionId() != null) {
            this.addTransportParameter(buffer, QuicConstants.TransportParameterId.retry_source_connection_id, this.params.getRetrySourceConnectionId());
        }
        if (this.discardTransportParameterSize != null) {
            this.addTransportParameter(buffer, (short)5950, new byte[this.discardTransportParameterSize.intValue()]);
        }
        int length = buffer.position();
        buffer.limit(length);
        int extensionsSize = length - 2 - 2;
        buffer.putShort(2, (short)extensionsSize);
        this.data = new byte[length];
        buffer.flip();
        buffer.get(this.data);
    }

    public QuicTransportParametersExtension parse(ByteBuffer buffer, Role senderRole, Logger log) throws DecodeErrorException {
        int extensionType = buffer.getShort() & 0xFFFF;
        if (!QuicTransportParametersExtension.isCodepoint(this.quicVersion, extensionType)) {
            throw new RuntimeException();
        }
        short extensionLength = buffer.getShort();
        int startPosition = buffer.position();
        log.debug("Transport parameters: ");
        while (buffer.position() - startPosition < extensionLength) {
            try {
                this.parseTransportParameter(buffer, senderRole, log);
            }
            catch (InvalidIntegerEncodingException e) {
                throw new DecodeErrorException("invalid integer encoding in transport parameter extension");
            }
        }
        int realSize = buffer.position() - startPosition;
        if (realSize != extensionLength) {
            throw new DecodeErrorException("inconsistent size in transport parameter extension");
        }
        return this;
    }

    void parseTransportParameter(ByteBuffer buffer, Role senderRol, Logger log) throws DecodeErrorException, InvalidIntegerEncodingException {
        long parameterId = VariableLengthInteger.parseLong(buffer);
        int size = VariableLengthInteger.parse(buffer);
        if (buffer.remaining() < size) {
            throw new DecodeErrorException("Invalid transport parameter extension");
        }
        int startPosition = buffer.position();
        if (parameterId == (long)QuicConstants.TransportParameterId.original_destination_connection_id.value) {
            if (senderRol != Role.Server) {
                throw new DecodeErrorException("server only parameter in transport parameter extension");
            }
            byte[] destinationCid = new byte[size];
            buffer.get(destinationCid);
            log.debug("- original destination connection id: ", destinationCid);
            this.params.setOriginalDestinationConnectionId(destinationCid);
        } else if (parameterId == (long)QuicConstants.TransportParameterId.max_idle_timeout.value) {
            long idleTimeout = VariableLengthInteger.parseLong(buffer);
            log.debug("- max idle timeout: " + idleTimeout);
            this.params.setMaxIdleTimeout(idleTimeout);
        } else if (parameterId == (long)QuicConstants.TransportParameterId.stateless_reset_token.value) {
            if (senderRol != Role.Server) {
                throw new DecodeErrorException("server only parameter in transport parameter extension");
            }
            byte[] resetToken = new byte[16];
            buffer.get(resetToken);
            log.debug("- stateless reset token: " + ByteUtils.bytesToHex(resetToken));
        } else if (parameterId == (long)QuicConstants.TransportParameterId.max_udp_payload_size.value) {
            int maxPacketSize = VariableLengthInteger.parse(buffer);
            log.debug("- max udp payload size: " + maxPacketSize);
            this.params.setMaxUdpPayloadSize(maxPacketSize);
        } else if (parameterId == (long)QuicConstants.TransportParameterId.initial_max_data.value) {
            long maxData = VariableLengthInteger.parseLong(buffer);
            log.debug("- initial max data: " + maxData);
            this.params.setInitialMaxData(maxData);
        } else if (parameterId == (long)QuicConstants.TransportParameterId.initial_max_stream_data_bidi_local.value) {
            int maxStreamDataBidiLocal = VariableLengthInteger.parse(buffer);
            log.debug("- initial max stream data bidi local: " + maxStreamDataBidiLocal);
            this.params.setInitialMaxStreamDataBidiLocal(maxStreamDataBidiLocal);
        } else if (parameterId == (long)QuicConstants.TransportParameterId.initial_max_stream_data_bidi_remote.value) {
            long maxStreamDataBidiRemote = VariableLengthInteger.parseLong(buffer);
            log.debug("- initial max stream data bidi remote: " + maxStreamDataBidiRemote);
            this.params.setInitialMaxStreamDataBidiRemote(maxStreamDataBidiRemote);
        } else if (parameterId == (long)QuicConstants.TransportParameterId.initial_max_stream_data_uni.value) {
            long maxStreamDataUni = VariableLengthInteger.parseLong(buffer);
            log.debug("- initial max stream data uni: " + maxStreamDataUni);
            this.params.setInitialMaxStreamDataUni(maxStreamDataUni);
        } else if (parameterId == (long)QuicConstants.TransportParameterId.initial_max_streams_bidi.value) {
            long maxBidiStreams = VariableLengthInteger.parseLong(buffer);
            log.debug("- initial max bidi streams: " + maxBidiStreams);
            this.params.setInitialMaxStreamsBidi(maxBidiStreams);
        } else if (parameterId == (long)QuicConstants.TransportParameterId.initial_max_streams_uni.value) {
            long maxUniStreams = VariableLengthInteger.parseLong(buffer);
            log.debug("- max uni streams: " + maxUniStreams);
            this.params.setInitialMaxStreamsUni(maxUniStreams);
        } else if (parameterId == (long)QuicConstants.TransportParameterId.ack_delay_exponent.value) {
            int ackDelayExponent = VariableLengthInteger.parse(buffer);
            log.debug("- ack delay exponent: " + ackDelayExponent);
            this.params.setAckDelayExponent(ackDelayExponent);
        } else if (parameterId == (long)QuicConstants.TransportParameterId.max_ack_delay.value) {
            int maxAckDelay = VariableLengthInteger.parse(buffer);
            log.debug("- max ack delay: " + maxAckDelay);
            this.params.setMaxAckDelay(maxAckDelay);
        } else if (parameterId == (long)QuicConstants.TransportParameterId.disable_active_migration.value) {
            log.debug("- disable migration");
            this.params.setDisableMigration(true);
        } else if (parameterId == (long)QuicConstants.TransportParameterId.preferred_address.value) {
            if (senderRol != Role.Server) {
                throw new DecodeErrorException("server only parameter in transport parameter extension");
            }
            this.parsePreferredAddress(buffer, log);
        } else if (parameterId == (long)QuicConstants.TransportParameterId.active_connection_id_limit.value) {
            int activeConnectionIdLimit = VariableLengthInteger.parse(buffer);
            log.debug("- active connection id limit: " + activeConnectionIdLimit);
            this.params.setActiveConnectionIdLimit(activeConnectionIdLimit);
        } else if (parameterId == (long)QuicConstants.TransportParameterId.initial_source_connection_id.value) {
            byte[] initialSourceCid = new byte[size];
            buffer.get(initialSourceCid);
            log.debug("- initial source connection id: " + initialSourceCid);
            this.params.setInitialSourceConnectionId(initialSourceCid);
        } else if (parameterId == (long)QuicConstants.TransportParameterId.retry_source_connection_id.value) {
            if (senderRol != Role.Server) {
                throw new DecodeErrorException("server only parameter in transport parameter extension");
            }
            byte[] retrySourceCid = new byte[size];
            buffer.get(retrySourceCid);
            log.debug("- retry source connection id: " + retrySourceCid);
            this.params.setRetrySourceConnectionId(retrySourceCid);
        } else {
            String extension = "";
            if (parameterId == 32L) {
                extension = "datagram";
            }
            if (parameterId == 64L) {
                extension = "multi-path";
            }
            if (parameterId == 4183L) {
                extension = "loss-bits";
            }
            if (parameterId == 5950L) {
                extension = "discard";
            }
            if (parameterId == 10930L) {
                extension = "grease-quic-bit";
            }
            if (parameterId == 29015L) {
                extension = "timestamp";
            }
            if (parameterId == 29016L) {
                extension = "timestamp";
            }
            if (parameterId == 29659L) {
                extension = "version-negotiation";
            }
            if (parameterId == 56858L) {
                extension = "delayed-ack";
            }
            if (parameterId == 4278378010L) {
                extension = "delayed-ack";
            }
            String msg = extension.isBlank() ? String.format("- unknown transport parameter 0x%04x, size %d", parameterId, size) : String.format("- unsupported transport parameter 0x%04x, size %d (%s)", parameterId, size, extension);
            log.warn(msg);
            buffer.get(new byte[size]);
        }
        int realSize = buffer.position() - startPosition;
        if (realSize != size) {
            throw new DecodeErrorException("inconsistent size in transport parameter");
        }
    }

    private void parsePreferredAddress(ByteBuffer buffer, Logger log) {
        try {
            TransportParameters.PreferredAddress preferredAddress = new TransportParameters.PreferredAddress();
            byte[] ip4 = new byte[4];
            buffer.get(ip4);
            if (!Bytes.allZero(ip4)) {
                preferredAddress.setIp4(InetAddress.getByAddress(ip4));
            }
            preferredAddress.setIp4Port(buffer.get() << 8 | buffer.get());
            byte[] ip6 = new byte[16];
            buffer.get(ip6);
            if (!Bytes.allZero(ip6)) {
                preferredAddress.setIp6(InetAddress.getByAddress(ip6));
            }
            preferredAddress.setIp6Port(buffer.get() << 8 | buffer.get());
            if (preferredAddress.getIp4() == null && preferredAddress.getIp6() == null) {
                throw new ProtocolError("Preferred address: no valid IP address");
            }
            byte connectionIdSize = buffer.get();
            preferredAddress.setConnectionId(buffer, connectionIdSize);
            preferredAddress.setStatelessResetToken(buffer, 16);
            this.params.setPreferredAddress(preferredAddress);
        }
        catch (UnknownHostException invalidIpAddressLength) {
            throw new RuntimeException();
        }
    }

    private void addTransportParameter(ByteBuffer buffer, QuicConstants.TransportParameterId id, long value) {
        this.addTransportParameter(buffer, id.value, value);
    }

    private void addTransportParameter(ByteBuffer buffer, short id, long value) {
        VariableLengthInteger.encode(id, buffer);
        buffer.mark();
        int encodedValueLength = VariableLengthInteger.encode(value, buffer);
        buffer.reset();
        VariableLengthInteger.encode(encodedValueLength, buffer);
        VariableLengthInteger.encode(value, buffer);
    }

    private void addTransportParameter(ByteBuffer buffer, QuicConstants.TransportParameterId id, byte[] value) {
        this.addTransportParameter(buffer, id.value, value);
    }

    private void addTransportParameter(ByteBuffer buffer, short id, byte[] value) {
        VariableLengthInteger.encode(id, buffer);
        VariableLengthInteger.encode(value.length, buffer);
        buffer.put(value);
    }

    public TransportParameters getTransportParameters() {
        return this.params;
    }
}

