/*
 * Decompiled with CFR 0.152.
 */
package net.luminis.quic.stream;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;
import net.luminis.quic.EncryptionLevel;
import net.luminis.quic.ImplementationError;
import net.luminis.quic.QuicClientConnectionImpl;
import net.luminis.quic.QuicConnectionImpl;
import net.luminis.quic.QuicConstants;
import net.luminis.quic.QuicStream;
import net.luminis.quic.Role;
import net.luminis.quic.TransportError;
import net.luminis.quic.Version;
import net.luminis.quic.frame.MaxStreamsFrame;
import net.luminis.quic.frame.QuicFrame;
import net.luminis.quic.frame.ResetStreamFrame;
import net.luminis.quic.frame.StopSendingFrame;
import net.luminis.quic.frame.StreamFrame;
import net.luminis.quic.log.Logger;
import net.luminis.quic.stream.EarlyDataStream;
import net.luminis.quic.stream.FlowControl;
import net.luminis.quic.stream.QuicStreamImpl;

public class StreamManager {
    private final Map<Integer, QuicStreamImpl> streams;
    private final Version quicVersion;
    private final QuicConnectionImpl connection;
    private FlowControl flowController;
    private final Role role;
    private final Logger log;
    private int maxOpenStreamIdUni;
    private int maxOpenStreamIdBidi;
    private int nextStreamId;
    private Consumer<QuicStream> peerInitiatedStreamCallback;
    private Long maxStreamsAcceptedByPeerBidi;
    private Long maxStreamsAcceptedByPeerUni;
    private final Semaphore openBidirectionalStreams;
    private final Semaphore openUnidirectionalStreams;
    private boolean maxOpenStreamsUniUpdateQueued;
    private boolean maxOpenStreamsBidiUpdateQueued;

    public StreamManager(QuicConnectionImpl quicConnection, Role role, Logger log, int maxOpenStreamsUni, int maxOpenStreamsBidi) {
        this.connection = quicConnection;
        this.role = role;
        this.log = log;
        this.maxOpenStreamIdUni = this.computeMaxStreamId(maxOpenStreamsUni, role.other(), false);
        this.maxOpenStreamIdBidi = this.computeMaxStreamId(maxOpenStreamsBidi, role.other(), true);
        this.quicVersion = Version.getDefault();
        this.streams = new ConcurrentHashMap<Integer, QuicStreamImpl>();
        this.openBidirectionalStreams = new Semaphore(0);
        this.openUnidirectionalStreams = new Semaphore(0);
    }

    private int computeMaxStreamId(int maxStreams, Role peerRole, boolean bidirectional) {
        int maxStreamId = maxStreams * 4;
        if (peerRole == Role.Server && bidirectional) {
            ++maxStreamId;
        }
        if (peerRole == Role.Client && !bidirectional) {
            maxStreamId += 2;
        }
        if (peerRole == Role.Client && bidirectional) {
            maxStreamId += 3;
        }
        return maxStreamId;
    }

    public QuicStream createStream(boolean bidirectional) {
        try {
            return this.createStream(bidirectional, 10000L, TimeUnit.DAYS);
        }
        catch (TimeoutException e) {
            throw new RuntimeException();
        }
    }

    public QuicStream createStream(boolean bidirectional, long timeout, TimeUnit timeoutUnit) throws TimeoutException {
        return this.createStream(bidirectional, timeout, timeoutUnit, (quicVersion, streamId, connection, flowController, logger) -> new QuicStreamImpl(quicVersion, streamId, connection, flowController, logger));
    }

    private QuicStreamImpl createStream(boolean bidirectional, long timeout, TimeUnit unit, QuicStreamSupplier streamFactory) throws TimeoutException {
        try {
            boolean acquired = bidirectional ? this.openBidirectionalStreams.tryAcquire(timeout, unit) : this.openUnidirectionalStreams.tryAcquire(timeout, unit);
            if (!acquired) {
                throw new TimeoutException();
            }
        }
        catch (InterruptedException e) {
            this.log.debug("blocked createStream operation is interrupted");
            throw new TimeoutException("operation interrupted");
        }
        int streamId = this.generateStreamId(bidirectional);
        QuicStreamImpl stream = streamFactory.apply(this.quicVersion, streamId, this.connection, this.flowController, this.log);
        this.streams.put(streamId, stream);
        return stream;
    }

    public EarlyDataStream createEarlyDataStream(boolean bidirectional) {
        try {
            return (EarlyDataStream)this.createStream(bidirectional, 0L, TimeUnit.MILLISECONDS, (quicVersion, streamId, connection, flowController, logger) -> new EarlyDataStream(quicVersion, streamId, (QuicClientConnectionImpl)connection, flowController, logger));
        }
        catch (TimeoutException e) {
            return null;
        }
    }

    private synchronized int generateStreamId(boolean bidirectional) {
        int id = (this.nextStreamId << 2) + (this.role == Role.Client ? 0 : 1);
        if (!bidirectional) {
            id += 2;
        }
        ++this.nextStreamId;
        return id;
    }

    public void setFlowController(FlowControl flowController) {
        this.flowController = flowController;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void process(StreamFrame frame) throws TransportError {
        int streamId = frame.getStreamId();
        QuicStreamImpl stream = this.streams.get(streamId);
        if (stream != null) {
            stream.add(frame);
            if (frame.isFinal() && this.isPeerInitiated(streamId)) {
                this.increaseMaxOpenStreams(streamId);
            }
        } else if (this.isPeerInitiated(streamId)) {
            StreamManager streamManager = this;
            synchronized (streamManager) {
                if (this.isUni(streamId) && streamId < this.maxOpenStreamIdUni || this.isBidi(streamId) && streamId < this.maxOpenStreamIdBidi) {
                    this.log.debug("Receiving data for peer-initiated stream " + streamId + " (#" + (streamId / 4 + 1) + " of this type)");
                    stream = new QuicStreamImpl(this.quicVersion, streamId, this.connection, this.flowController, this.log);
                    this.streams.put(streamId, stream);
                    stream.add(frame);
                    if (this.peerInitiatedStreamCallback != null) {
                        this.peerInitiatedStreamCallback.accept(stream);
                    }
                    if (frame.isFinal()) {
                        this.increaseMaxOpenStreams(streamId);
                    }
                } else {
                    throw new TransportError(QuicConstants.TransportErrorCode.STREAM_LIMIT_ERROR);
                }
            }
        } else {
            this.log.error("Receiving frame for non-existent stream " + streamId);
        }
    }

    public void process(StopSendingFrame stopSendingFrame) {
        QuicStreamImpl stream = this.streams.get(stopSendingFrame.getStreamId());
        if (stream != null) {
            stream.resetStream(stopSendingFrame.getErrorCode());
        }
    }

    public void process(ResetStreamFrame resetStreamFrame) {
        QuicStreamImpl stream = this.streams.get(resetStreamFrame.getStreamId());
        if (stream != null) {
            stream.terminateStream(resetStreamFrame.getErrorCode(), resetStreamFrame.getFinalSize());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void increaseMaxOpenStreams(int streamId) {
        StreamManager streamManager = this;
        synchronized (streamManager) {
            if (this.isUni(streamId)) {
                this.maxOpenStreamIdUni += 4;
                if (!this.maxOpenStreamsUniUpdateQueued) {
                    this.connection.send(this::createMaxStreamsUpdateUni, 9, EncryptionLevel.App, this::retransmitMaxStreams);
                    this.maxOpenStreamsUniUpdateQueued = true;
                }
            } else {
                this.maxOpenStreamIdBidi += 4;
                if (!this.maxOpenStreamsBidiUpdateQueued) {
                    this.connection.send(this::createMaxStreamsUpdateBidi, 9, EncryptionLevel.App, this::retransmitMaxStreams);
                    this.maxOpenStreamsBidiUpdateQueued = true;
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private QuicFrame createMaxStreamsUpdateUni(int maxSize) {
        if (maxSize < 9) {
            throw new ImplementationError();
        }
        StreamManager streamManager = this;
        synchronized (streamManager) {
            this.maxOpenStreamsUniUpdateQueued = false;
        }
        return new MaxStreamsFrame(this.maxOpenStreamIdUni / 4, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private QuicFrame createMaxStreamsUpdateBidi(int maxSize) {
        if (maxSize < 9) {
            throw new ImplementationError();
        }
        StreamManager streamManager = this;
        synchronized (streamManager) {
            this.maxOpenStreamsBidiUpdateQueued = false;
        }
        return new MaxStreamsFrame(this.maxOpenStreamIdBidi / 4, true);
    }

    void retransmitMaxStreams(QuicFrame frame) {
        MaxStreamsFrame lostFrame = (MaxStreamsFrame)frame;
        if (lostFrame.isAppliesToBidirectional()) {
            this.connection.send(this.createMaxStreamsUpdateBidi(Integer.MAX_VALUE), this::retransmitMaxStreams);
        } else {
            this.connection.send(this.createMaxStreamsUpdateUni(Integer.MAX_VALUE), this::retransmitMaxStreams);
        }
    }

    private boolean isPeerInitiated(int streamId) {
        return streamId % 2 == (this.role == Role.Client ? 1 : 0);
    }

    private boolean isUni(int streamId) {
        return streamId % 4 > 1;
    }

    private boolean isBidi(int streamId) {
        return streamId % 4 < 2;
    }

    public synchronized void process(MaxStreamsFrame frame) {
        if (frame.isAppliesToBidirectional()) {
            if (frame.getMaxStreams() > this.maxStreamsAcceptedByPeerBidi) {
                int increment = (int)(frame.getMaxStreams() - this.maxStreamsAcceptedByPeerBidi);
                this.log.debug("increased max bidirectional streams with " + increment + " to " + frame.getMaxStreams());
                this.maxStreamsAcceptedByPeerBidi = frame.getMaxStreams();
                this.openBidirectionalStreams.release(increment);
            }
        } else if (frame.getMaxStreams() > this.maxStreamsAcceptedByPeerUni) {
            int increment = (int)(frame.getMaxStreams() - this.maxStreamsAcceptedByPeerUni);
            this.log.debug("increased max unidirectional streams with " + increment + " to " + frame.getMaxStreams());
            this.maxStreamsAcceptedByPeerUni = frame.getMaxStreams();
            this.openUnidirectionalStreams.release(increment);
        }
    }

    public void abortAll() {
        this.streams.values().stream().forEach(s -> s.abort());
    }

    public synchronized void setPeerInitiatedStreamCallback(Consumer<QuicStream> streamProcessor) {
        this.peerInitiatedStreamCallback = streamProcessor;
    }

    public synchronized void setInitialMaxStreamsBidi(long initialMaxStreamsBidi) {
        if (this.maxStreamsAcceptedByPeerBidi == null || initialMaxStreamsBidi >= this.maxStreamsAcceptedByPeerBidi) {
            this.log.debug("Initial max bidirectional stream: " + initialMaxStreamsBidi);
            this.maxStreamsAcceptedByPeerBidi = initialMaxStreamsBidi;
            if (initialMaxStreamsBidi > Integer.MAX_VALUE) {
                this.log.error("Server initial max streams bidirectional is larger than supported; limiting to 2147483647");
                initialMaxStreamsBidi = Integer.MAX_VALUE;
            }
            this.openBidirectionalStreams.release((int)initialMaxStreamsBidi);
        } else {
            this.log.error("Attempt to reduce value of initial_max_streams_bidi from " + this.maxStreamsAcceptedByPeerBidi + " to " + initialMaxStreamsBidi + "; ignoring.");
        }
    }

    public synchronized void setInitialMaxStreamsUni(long initialMaxStreamsUni) {
        if (this.maxStreamsAcceptedByPeerUni == null || initialMaxStreamsUni >= this.maxStreamsAcceptedByPeerUni) {
            this.log.debug("Initial max unidirectional stream: " + initialMaxStreamsUni);
            this.maxStreamsAcceptedByPeerUni = initialMaxStreamsUni;
            if (initialMaxStreamsUni > Integer.MAX_VALUE) {
                this.log.error("Server initial max streams unidirectional is larger than supported; limiting to 2147483647");
                initialMaxStreamsUni = Integer.MAX_VALUE;
            }
            this.openUnidirectionalStreams.release((int)initialMaxStreamsUni);
        } else {
            this.log.error("Attempt to reduce value of initial_max_streams_uni from " + this.maxStreamsAcceptedByPeerUni + " to " + initialMaxStreamsUni + "; ignoring.");
        }
    }

    public synchronized long getMaxBidirectionalStreams() {
        return this.maxStreamsAcceptedByPeerBidi;
    }

    public synchronized long getMaxUnirectionalStreams() {
        return this.maxStreamsAcceptedByPeerUni;
    }

    static interface QuicStreamSupplier {
        public QuicStreamImpl apply(Version var1, int var2, QuicConnectionImpl var3, FlowControl var4, Logger var5);
    }
}

