/*
 * Decompiled with CFR 0.152.
 */
package net.luminis.quic.recovery;

import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import net.luminis.quic.cc.CongestionController;
import net.luminis.quic.frame.AckFrame;
import net.luminis.quic.packet.PacketInfo;
import net.luminis.quic.packet.QuicPacket;
import net.luminis.quic.recovery.PacketStatus;
import net.luminis.quic.recovery.RecoveryManager;
import net.luminis.quic.recovery.RttEstimator;

public class LossDetector {
    private final RecoveryManager recoveryManager;
    private final RttEstimator rttEstimater;
    private final CongestionController congestionController;
    private float kTimeThreshold = 1.125f;
    private int kPacketThreshold = 3;
    private final Map<Long, PacketStatus> packetSentLog;
    private final AtomicInteger ackElicitingInFlight;
    private volatile long largestAcked = -1L;
    private volatile long lost;
    private volatile Instant lossTime;
    private volatile Instant lastAckElicitingSent;
    private volatile boolean isReset;

    public LossDetector(RecoveryManager recoveryManager, RttEstimator rttEstimator, CongestionController congestionController) {
        this.recoveryManager = recoveryManager;
        this.rttEstimater = rttEstimator;
        this.congestionController = congestionController;
        this.ackElicitingInFlight = new AtomicInteger();
        this.packetSentLog = new ConcurrentHashMap<Long, PacketStatus>();
    }

    public synchronized void packetSent(QuicPacket packet, Instant sent, Consumer<QuicPacket> lostPacketCallback) {
        if (this.isReset) {
            return;
        }
        if (packet.isInflightPacket()) {
            this.congestionController.registerInFlight(packet);
        }
        if (packet.isAckEliciting()) {
            this.ackElicitingInFlight.getAndAdd(1);
            this.lastAckElicitingSent = sent;
        }
        this.packetSentLog.put(packet.getPacketNumber(), new PacketStatus(sent, packet, lostPacketCallback));
    }

    public void onAckReceived(AckFrame ackFrame, Instant timeReceived) {
        if (this.isReset) {
            return;
        }
        this.largestAcked = Long.max(this.largestAcked, ackFrame.getLargestAcknowledged());
        List<PacketStatus> newlyAcked = ackFrame.getAckedPacketNumbers().stream().filter(pn -> this.packetSentLog.containsKey(pn) && !this.packetSentLog.get(pn).acked()).map(pn -> this.packetSentLog.get(pn)).filter(packetStatus -> packetStatus != null).filter(packetStatus -> packetStatus.setAcked()).collect(Collectors.toList());
        int ackedAckEliciting = (int)newlyAcked.stream().filter(packetStatus -> packetStatus.packet().isAckEliciting()).count();
        assert (ackedAckEliciting <= this.ackElicitingInFlight.get());
        this.ackElicitingInFlight.getAndAdd(-1 * ackedAckEliciting);
        this.congestionController.registerAcked(this.filterInFlight(newlyAcked));
        this.detectLostPackets();
        this.recoveryManager.setLossDetectionTimer();
        this.rttEstimater.ackReceived(ackFrame, timeReceived, newlyAcked);
    }

    public synchronized void reset() {
        List inflightPackets = this.packetSentLog.values().stream().filter(packet -> packet.inFlight()).filter(packetStatus -> packetStatus.setLost()).collect(Collectors.toList());
        this.congestionController.discard(inflightPackets);
        this.ackElicitingInFlight.set(0);
        this.packetSentLog.clear();
        this.lossTime = null;
        this.lastAckElicitingSent = null;
        this.isReset = true;
    }

    void detectLostPackets() {
        Optional<Instant> earliestSentTime;
        if (this.isReset) {
            return;
        }
        int lossDelay = (int)(this.kTimeThreshold * (float)Integer.max(this.rttEstimater.getSmoothedRtt(), this.rttEstimater.getLatestRtt()));
        assert (lossDelay > 0);
        Instant lostSendTime = Instant.now().minusMillis(lossDelay);
        List<PacketStatus> lostPackets = this.packetSentLog.values().stream().filter(p -> p.inFlight()).filter(p -> this.pnTooOld((PacketStatus)p) || this.sentTimeTooLongAgo((PacketStatus)p, lostSendTime)).filter(p -> !p.packet().isAckOnly()).collect(Collectors.toList());
        if (!lostPackets.isEmpty()) {
            this.declareLost(lostPackets);
        }
        this.lossTime = (earliestSentTime = this.packetSentLog.values().stream().filter(p -> p.inFlight()).filter(p -> p.packet().getPacketNumber() <= this.largestAcked).filter(p -> !p.packet().isAckOnly()).map(p -> p.timeSent()).min(Instant::compareTo)).isPresent() && earliestSentTime.get().isAfter(lostSendTime) ? earliestSentTime.get().plusMillis(lossDelay) : null;
    }

    Instant getLossTime() {
        return this.lossTime;
    }

    Instant getLastAckElicitingSent() {
        return this.lastAckElicitingSent;
    }

    boolean ackElicitingInFlight() {
        int actualAckElicitingInFlight = this.ackElicitingInFlight.get();
        assert (actualAckElicitingInFlight >= 0);
        return actualAckElicitingInFlight != 0;
    }

    List<QuicPacket> unAcked() {
        return this.packetSentLog.values().stream().filter(p -> p.inFlight()).filter(p -> !p.packet().isAckOnly()).map(p -> p.packet()).collect(Collectors.toList());
    }

    List<PacketInfo> getInFlight() {
        return this.packetSentLog.values().stream().filter(p -> !p.packet().isAckOnly()).filter(p -> p.inFlight()).collect(Collectors.toList());
    }

    private boolean pnTooOld(PacketStatus p) {
        return p.packet().getPacketNumber() <= this.largestAcked - (long)this.kPacketThreshold;
    }

    private boolean sentTimeTooLongAgo(PacketStatus p, Instant lostSendTime) {
        return p.packet().getPacketNumber() <= this.largestAcked && p.timeSent().isBefore(lostSendTime);
    }

    private void declareLost(List<PacketStatus> lostPacketsInfo) {
        lostPacketsInfo = lostPacketsInfo.stream().filter(packetStatus -> packetStatus.setLost()).collect(Collectors.toList());
        int lostAckEliciting = (int)lostPacketsInfo.stream().filter(packetStatus -> packetStatus.packet().isAckEliciting()).count();
        assert (lostAckEliciting <= this.ackElicitingInFlight.get());
        this.ackElicitingInFlight.getAndAdd(-1 * lostAckEliciting);
        lostPacketsInfo.stream().forEach(packetStatus -> {
            packetStatus.lostPacketCallback().accept(packetStatus.packet());
            ++this.lost;
        });
        this.congestionController.registerLost(this.filterInFlight(lostPacketsInfo));
    }

    private List<PacketStatus> filterInFlight(List<PacketStatus> packets) {
        return packets.stream().filter(packetInfo -> packetInfo.packet().isInflightPacket()).collect(Collectors.toList());
    }

    public long getLost() {
        return this.lost;
    }

    public boolean noAckedReceived() {
        return this.largestAcked < 0L;
    }
}

