package dev.jfr4jdbc;

import dev.jfr4jdbc.interceptor.*;
import dev.jfr4jdbc.internal.ConnectionInfo;
import dev.jfr4jdbc.internal.OperationInfo;

import java.sql.*;

abstract public class JfrStatement42 implements Statement {

    protected final InterceptorFactory interceptorFactory;
    protected final Statement jdbcStatement;

    private final ConnectionInfo connectionInfo;

    private final OperationInfo operationInfo;

    public OperationInfo getOperationInfo() {
        return this.operationInfo;
    }

    private StringBuilder batchSql;

    protected JfrStatement42(Statement s) {
        this(s, InterceptorManager.getDefaultInterceptorFactory(), new ConnectionInfo(null, 0, 0), new OperationInfo(0));
    }

    protected JfrStatement42(Statement s, InterceptorFactory factory) {
        this(s, factory, new ConnectionInfo(null, 0, 0), new OperationInfo(0));
    }

    protected JfrStatement42(Statement s, InterceptorFactory factory, ConnectionInfo connectionInfo, OperationInfo operationInfo) {
        super();
        this.jdbcStatement = s;
        this.interceptorFactory = factory;
        this.connectionInfo = connectionInfo;
        this.operationInfo = operationInfo;
    }

    protected StatementContext createContext(String inquiry, boolean isPrepared) {

        StatementContext context = new StatementContext(this.jdbcStatement, this.connectionInfo, this.operationInfo, inquiry, isPrepared);
        try {
            if (this.jdbcStatement != null) {
                // Add Statement Info.
                context.setStatementPoolable(this.jdbcStatement.isPoolable());
                context.setStatementClosed(this.jdbcStatement.isClosed());

                // Add Connection Info.
                Connection con = this.jdbcStatement.getConnection();
                if (con != null) {
                    context.setConnection(con);
                    context.setAutoCommitted(con.getAutoCommit());
                }
            }
        } catch (SQLException e) {
        }

        return context;
    }

    protected JfrResultSet createResultSet(ResultSet rs) {
        return new JfrResultSet(rs, interceptorFactory, this.connectionInfo, this.operationInfo);
    }

    @Override
    public ResultSet executeQuery(String sql) throws SQLException {

        Interceptor<StatementContext> interceptor = interceptorFactory.createStatementInterceptor();
        StatementContext context = this.createContext(sql, false);

        ResultSet rs = null;
        try {
            interceptor.preInvoke(context);
            rs = this.jdbcStatement.executeQuery(sql);
        } catch (SQLException | RuntimeException e) {
            context.setException(e);
            throw e;
        } finally {
            interceptor.postInvoke(context);
        }

        return this.createResultSet(rs);
    }

    @Override
    public ResultSet getResultSet() throws SQLException {

        Interceptor<StatementContext> interceptor = interceptorFactory.createStatementInterceptor();
        StatementContext context = this.createContext("getResultSet", false);

        ResultSet rs;
        try {
            interceptor.preInvoke(context);
            rs = this.jdbcStatement.getResultSet();
        } catch (SQLException | RuntimeException e) {
            context.setException(e);
            throw e;
        } finally {
            interceptor.postInvoke(context);
        }

        return this.createResultSet(rs);
    }

    @Override
    public ResultSet getGeneratedKeys() throws SQLException {

        Interceptor<StatementContext> interceptor = interceptorFactory.createStatementInterceptor();
        StatementContext context = this.createContext("getGeneratedKeys", false);

        ResultSet rs;
        try {
            interceptor.preInvoke(context);
            rs = this.jdbcStatement.getGeneratedKeys();
        } catch (SQLException | RuntimeException e) {
            context.setException(e);
            throw e;
        } finally {
            interceptor.postInvoke(context);
        }

        return this.createResultSet(rs);
    }

    @Override
    public int executeUpdate(String sql) throws SQLException {

        Interceptor<StatementContext> interceptor = interceptorFactory.createStatementInterceptor();
        StatementContext context = this.createContext(sql, false);

        int result;
        try {
            interceptor.preInvoke(context);
            result = this.jdbcStatement.executeUpdate(sql);
        } catch (SQLException | RuntimeException e) {
            context.setException(e);
            throw e;
        } finally {
            interceptor.postInvoke(context);
        }

        return result;
    }

    @Override
    public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException {

        Interceptor<StatementContext> interceptor = interceptorFactory.createStatementInterceptor();
        StatementContext context = this.createContext(sql, false);

        int result;
        try {
            interceptor.preInvoke(context);
            result = this.jdbcStatement.executeUpdate(sql, autoGeneratedKeys);
        } catch (SQLException | RuntimeException e) {
            context.setException(e);
            throw e;
        } finally {
            interceptor.postInvoke(context);
        }

        return result;
    }

    @Override
    public int executeUpdate(String sql, int[] columnIndexes) throws SQLException {

        Interceptor<StatementContext> interceptor = interceptorFactory.createStatementInterceptor();
        StatementContext context = this.createContext(sql, false);

        int result;
        try {
            interceptor.preInvoke(context);
            result = this.jdbcStatement.executeUpdate(sql, columnIndexes);
        } catch (SQLException | RuntimeException e) {
            context.setException(e);
            throw e;
        } finally {
            interceptor.postInvoke(context);
        }

        return result;
    }

    @Override
    public int executeUpdate(String sql, String[] columnNames) throws SQLException {

        Interceptor<StatementContext> interceptor = interceptorFactory.createStatementInterceptor();
        StatementContext context = this.createContext(sql, false);

        int result;
        try {
            interceptor.preInvoke(context);
            result = this.jdbcStatement.executeUpdate(sql, columnNames);
        } catch (SQLException | RuntimeException e) {
            context.setException(e);
            throw e;
        } finally {
            interceptor.postInvoke(context);
        }

        return result;
    }

    @Override
    public boolean execute(String sql) throws SQLException {

        Interceptor<StatementContext> interceptor = interceptorFactory.createStatementInterceptor();
        StatementContext context = this.createContext(sql, false);

        boolean result;
        try {
            interceptor.preInvoke(context);
            result = this.jdbcStatement.execute(sql);
        } catch (SQLException | RuntimeException e) {
            context.setException(e);
            throw e;
        } finally {
            interceptor.postInvoke(context);
        }

        return result;
    }

    @Override
    public boolean execute(String sql, int autoGeneratedKeys) throws SQLException {

        Interceptor<StatementContext> interceptor = interceptorFactory.createStatementInterceptor();
        StatementContext context = this.createContext(sql, false);

        boolean result;
        try {
            interceptor.preInvoke(context);
            result = this.jdbcStatement.execute(sql, autoGeneratedKeys);
        } catch (SQLException | RuntimeException e) {
            context.setException(e);
            throw e;
        } finally {
            interceptor.postInvoke(context);
        }

        return result;
    }

    @Override
    public boolean execute(String sql, int[] columnIndexes) throws SQLException {

        Interceptor<StatementContext> interceptor = interceptorFactory.createStatementInterceptor();
        StatementContext context = this.createContext(sql, false);

        boolean result;
        try {
            interceptor.preInvoke(context);
            result = this.jdbcStatement.execute(sql, columnIndexes);
        } catch (SQLException | RuntimeException e) {
            context.setException(e);
            throw e;
        } finally {
            interceptor.postInvoke(context);
        }

        return result;
    }

    @Override
    public boolean execute(String sql, String[] columnNames) throws SQLException {

        Interceptor<StatementContext> interceptor = interceptorFactory.createStatementInterceptor();
        StatementContext context = this.createContext(sql, false);

        boolean result;
        try {
            interceptor.preInvoke(context);
            result = this.jdbcStatement.execute(sql, columnNames);
        } catch (SQLException | RuntimeException e) {
            context.setException(e);
            throw e;
        } finally {
            interceptor.postInvoke(context);
        }

        return result;
    }

    @Override
    public int[] executeBatch() throws SQLException {

        String sql = (this.batchSql == null) ? "" : this.batchSql.toString();

        Interceptor<StatementContext> interceptor = interceptorFactory.createStatementInterceptor();
        StatementContext context = this.createContext(sql, false);

        int[] result;
        try {
            interceptor.preInvoke(context);
            result = this.jdbcStatement.executeBatch();
        } catch (SQLException | RuntimeException e) {
            context.setException(e);
            throw e;
        } finally {
            interceptor.postInvoke(context);
        }

        return result;
    }

    @Override
    public void cancel() throws SQLException {

        Interceptor<CancelContext> interceptor = interceptorFactory.createCancelInterceptor();
        CancelContext context = new CancelContext(this.getConnection(), this, this.connectionInfo, this.operationInfo);

        try {
            interceptor.preInvoke(context);
            this.jdbcStatement.cancel();
        } catch (SQLException | RuntimeException e) {
            context.setException(e);
            throw e;
        } finally {
            interceptor.postInvoke(context);
        }
    }

    @Override
    public void addBatch(String sql) throws SQLException {

        if (this.batchSql == null) {
            this.batchSql = new StringBuilder();
        }
        this.batchSql.append(sql);
        this.batchSql.append(";");

        this.jdbcStatement.addBatch(sql);
    }

    @Override
    public void clearBatch() throws SQLException {
        this.batchSql = null;
        this.jdbcStatement.clearBatch();
    }

    @Override
    public <T> T unwrap(Class<T> iface) throws SQLException {
        return this.jdbcStatement.unwrap(iface);
    }

    @Override
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        return this.jdbcStatement.isWrapperFor(iface);
    }

    @Override
    public void close() throws SQLException {
        this.jdbcStatement.close();
    }

    @Override
    public int getMaxFieldSize() throws SQLException {
        return this.jdbcStatement.getMaxFieldSize();
    }

    @Override
    public void setMaxFieldSize(int max) throws SQLException {
        this.jdbcStatement.setMaxFieldSize(max);
    }

    @Override
    public int getMaxRows() throws SQLException {
        return this.jdbcStatement.getMaxRows();
    }

    @Override
    public void setMaxRows(int max) throws SQLException {
        this.jdbcStatement.setMaxRows(max);
    }

    @Override
    public void setEscapeProcessing(boolean enable) throws SQLException {
        this.jdbcStatement.setEscapeProcessing(enable);
    }

    @Override
    public int getQueryTimeout() throws SQLException {
        return this.jdbcStatement.getQueryTimeout();
    }

    @Override
    public void setQueryTimeout(int seconds) throws SQLException {

        this.jdbcStatement.setQueryTimeout(seconds);
    }

    @Override
    public SQLWarning getWarnings() throws SQLException {

        return this.jdbcStatement.getWarnings();
    }

    @Override
    public void clearWarnings() throws SQLException {

        this.jdbcStatement.clearWarnings();
    }

    @Override
    public void setCursorName(String name) throws SQLException {

        this.jdbcStatement.setCursorName(name);
    }

    @Override
    public int getUpdateCount() throws SQLException {

        return this.jdbcStatement.getUpdateCount();
    }

    @Override
    public boolean getMoreResults() throws SQLException {
        return this.jdbcStatement.getMoreResults();
    }

    @Override
    public void setFetchDirection(int direction) throws SQLException {

        this.jdbcStatement.setFetchDirection(direction);
    }

    @Override
    public int getFetchDirection() throws SQLException {

        return this.jdbcStatement.getFetchDirection();
    }

    @Override
    public void setFetchSize(int rows) throws SQLException {
        this.jdbcStatement.setFetchSize(rows);
    }

    @Override
    public int getFetchSize() throws SQLException {
        return this.jdbcStatement.getFetchSize();
    }

    @Override
    public int getResultSetConcurrency() throws SQLException {
        return this.jdbcStatement.getResultSetConcurrency();
    }

    @Override
    public int getResultSetType() throws SQLException {

        return this.jdbcStatement.getResultSetType();
    }

    @Override
    public Connection getConnection() throws SQLException {
        return this.jdbcStatement.getConnection();
    }

    @Override
    public boolean getMoreResults(int current) throws SQLException {
        return this.jdbcStatement.getMoreResults(current);
    }

    @Override
    public int getResultSetHoldability() throws SQLException {

        return this.jdbcStatement.getResultSetHoldability();
    }

    @Override
    public boolean isClosed() throws SQLException {
        return this.jdbcStatement.isClosed();
    }

    @Override
    public void setPoolable(boolean poolable) throws SQLException {
        this.jdbcStatement.setPoolable(poolable);
    }

    @Override
    public boolean isPoolable() throws SQLException {
        return this.jdbcStatement.isPoolable();
    }

    @Override
    public void closeOnCompletion() throws SQLException {
        this.jdbcStatement.closeOnCompletion();
    }

    @Override
    public boolean isCloseOnCompletion() throws SQLException {
        return this.jdbcStatement.isCloseOnCompletion();
    }

    @Override
    public long getLargeUpdateCount() throws SQLException {
        return this.jdbcStatement.getLargeUpdateCount();
    }

    @Override
    public void setLargeMaxRows(long max) throws SQLException {
        this.jdbcStatement.setLargeMaxRows(max);
    }

    @Override
    public long getLargeMaxRows() throws SQLException {
        return this.jdbcStatement.getLargeMaxRows();
    }

    @Override
    public long[] executeLargeBatch() throws SQLException {
        return this.jdbcStatement.executeLargeBatch();
    }

    @Override
    public long executeLargeUpdate(String sql) throws SQLException {
        return this.jdbcStatement.executeLargeUpdate(sql);
    }

    @Override
    public long executeLargeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
        return this.jdbcStatement.executeLargeUpdate(sql, autoGeneratedKeys);
    }

    @Override
    public long executeLargeUpdate(String sql, int[] columnIndexes) throws SQLException {
        return this.jdbcStatement.executeLargeUpdate(sql, columnIndexes);
    }

    @Override
    public long executeLargeUpdate(String sql, String[] columnNames) throws SQLException {
        return this.jdbcStatement.executeLargeUpdate(sql, columnNames);
    }
}
