package dev.inmo.jsuikit.elements

import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import dev.inmo.jsuikit.modifiers.*
import org.jetbrains.compose.web.dom.*
import org.jetbrains.compose.web.renderComposableInBody
import org.w3c.dom.HTMLDivElement
import org.w3c.dom.MutationObserver
import org.w3c.dom.MutationObserverInit
import kotlin.random.Random
import kotlin.random.nextUInt

@Composable
fun Dialog(
    vararg modifiers: UIKitModifier,
    attributesCustomizer: AttrBuilderContext<HTMLDivElement> = {},
    onHidden: ((HTMLDivElement) -> Unit)? = null,
    onShown: ((HTMLDivElement) -> Unit)? = null,
    dialogAttrsBuilder: AttrBuilderContext<HTMLDivElement>? = null,
    headerAttrsBuilder: AttrBuilderContext<HTMLDivElement>? = null,
    headerBuilder: ContentBuilder<HTMLDivElement>? = null,
    afterHeaderBuilder: ContentBuilder<HTMLDivElement>? = null,
    beforeFooterBuilder: ContentBuilder<HTMLDivElement>? = null,
    footerAttrsBuilder: AttrBuilderContext<HTMLDivElement>? = null,
    footerBuilder: ContentBuilder<HTMLDivElement>? = null,
    bodyAttrsBuilder: AttrBuilderContext<HTMLDivElement>? = null,
    autoShow: Boolean = true,
    removeOnHide: Boolean = true,
    bodyBuilder: ContentBuilder<HTMLDivElement> = {}
) {
    val drawDiv = remember { mutableStateOf(true) }
    val composition = remember {
        renderComposableInBody {
            Div(
                {
                    if (modifiers.none { it is UIKitModal.WithCustomAttributes }) {
                        include(UIKitModal)
                    }
                    id("dialog${Random.nextUInt()}")
                    include(*modifiers)

                    attributesCustomizer()
                }
            ) {
                DisposableEffect(true) {
                    val htmlElement = scopeElement

                    if (autoShow) {
                        UIKit.modal(htmlElement).show()
                    }

                    if (onHidden != null || removeOnHide) {
                        htmlElement.addEventListener("hidden", {
                            onHidden ?.invoke(htmlElement)

                            if (removeOnHide) {
                                htmlElement.remove()
                            }
                        })
                    }

                    onShown ?.let {
                        htmlElement.addEventListener("shown", {
                            onShown(htmlElement)
                        })
                    }


                    onDispose {
                        drawDiv.value = false
                    }
                }

                Div(
                    {
                        include(UIKitModal.Dialog)
                        dialogAttrsBuilder ?.let { it() } ?: include(UIKitMargin.Auto.Vertical)
                    }
                ) {
                    headerBuilder ?.let {
                        Div(
                            {
                                include(UIKitModal.Header)
                                headerAttrsBuilder ?.let { it() }
                            }
                        ) {
                            it()
                        }
                    }
                    afterHeaderBuilder ?.let { it() }
                    Div(
                        {
                            include(UIKitModal.Body)
                            bodyAttrsBuilder ?.let { it() }
                        }
                    ) {
                        bodyBuilder()
                    }
                    beforeFooterBuilder ?.let { it() }
                    footerBuilder ?.let {
                        Div(
                            {
                                include(UIKitModal.Footer)
                                footerAttrsBuilder ?.let { it() } ?: include(UIKitText.Alignment.Horizontal.Right)
                            }
                        ) {
                            it()
                        }
                    }
                }
            }
        }
    }

    if (drawDiv.value) {
        Div({
            hidden()
            ref {
                onDispose {
                    composition.dispose()
                }
            }
        })
    } else {
        runCatching { composition.dispose() }
    }
}

@Composable
fun Dialog(
    title: String,
    vararg modifiers: UIKitModifier,
    attributesCustomizer: AttrBuilderContext<HTMLDivElement> = {},
    onHidden: ((HTMLDivElement) -> Unit)? = null,
    onShown: ((HTMLDivElement) -> Unit)? = null,
    dialogAttrsBuilder: AttrBuilderContext<HTMLDivElement>? = null,
    headerAttrsBuilder: AttrBuilderContext<HTMLDivElement>? = null,
    headerBuilder: ContentBuilder<HTMLDivElement>? = null,
    afterHeaderBuilder: ContentBuilder<HTMLDivElement>? = null,
    beforeFooterBuilder: ContentBuilder<HTMLDivElement>? = null,
    footerAttrsBuilder: AttrBuilderContext<HTMLDivElement>? = null,
    footerBuilder: ContentBuilder<HTMLDivElement>? = null,
    bodyAttrsBuilder: AttrBuilderContext<HTMLDivElement>? = null,
    autoShow: Boolean = true,
    removeOnHide: Boolean = true,
    bodyBuilder: ContentBuilder<HTMLDivElement> = {}
) = Dialog(
    modifiers = modifiers,
    attributesCustomizer = attributesCustomizer,
    onHidden = onHidden,
    onShown = onShown,
    dialogAttrsBuilder = dialogAttrsBuilder,
    headerAttrsBuilder = headerAttrsBuilder,
    headerBuilder = {
        H2({ include(UIKitModal.Title) }) {
            Text(title)
        }
        headerBuilder ?.invoke(this)
    },
    afterHeaderBuilder = afterHeaderBuilder,
    beforeFooterBuilder = beforeFooterBuilder,
    footerAttrsBuilder = footerAttrsBuilder,
    footerBuilder = footerBuilder,
    bodyAttrsBuilder = bodyAttrsBuilder,
    autoShow = autoShow,
    removeOnHide = removeOnHide,
    bodyBuilder = bodyBuilder
)
