/*
 * Decompiled with CFR 0.152.
 */
package net.hollowcube.posthog;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonNull;
import com.google.gson.JsonObject;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.net.http.HttpTimeoutException;
import java.time.Duration;
import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.BiFunction;
import net.hollowcube.posthog.EventQueue;
import net.hollowcube.posthog.FeatureFlagContext;
import net.hollowcube.posthog.FeatureFlagEvaluator;
import net.hollowcube.posthog.FeatureFlagState;
import net.hollowcube.posthog.FeatureFlagStates;
import net.hollowcube.posthog.FeatureFlagsResponse;
import net.hollowcube.posthog.PostHogClient;
import net.hollowcube.posthog.PostHogNames;
import net.hollowcube.posthog.Timer;
import org.jetbrains.annotations.Blocking;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class PostHogClientImpl
implements PostHogClient {
    private static final String DEFAULT_LIBRARY_NAME = "github.com/hollow-cube/posthog-java";
    private static final String DEFAULT_LIBRARY_VERSION = "1.0.0";
    private static final String USER_AGENT = String.format("%s/%s", "github.com/hollow-cube/posthog-java", "1.0.0");
    private static final int STACKTRACE_FRAME_LIMIT = 100;
    private static final Logger log = LoggerFactory.getLogger(PostHogClientImpl.class);
    private final HttpClient httpClient = HttpClient.newHttpClient();
    private final EventQueue queue;
    private final Timer featureFlagFetchTimer;
    private final Gson gson;
    private final String endpoint;
    private final String projectApiKey;
    private final String personalApiKey;
    private final JsonObject defaultEventProperties;
    private final Duration eventBatchTimeout;
    private Map<String, FeatureFlagsResponse.Flag> featureFlags = null;
    private final Map<String, Object> recentlyCapturedFeatureFlags = new ConcurrentHashMap<String, Object>();
    private final boolean allowRemoteFeatureFlagEvaluation;
    private final boolean sendFeatureFlagEvents;
    private final Duration featureFlagsRequestTimeout;
    private final BiFunction<Throwable, JsonObject, Boolean> exceptionMiddleware;

    PostHogClientImpl(@NotNull Gson gson, @NotNull String endpoint, @NotNull String projectApiKey, @Nullable String personalApiKey, @NotNull Duration flushInterval, int maxBatchSize, @NotNull Map<String, Object> defaultEventProperties, @NotNull Duration eventBatchTimeout, boolean allowRemoteFeatureFlagEvaluation, boolean sendFeatureFlagEvents, @NotNull Duration featureFlagsPollingInterval, @NotNull Duration featureFlagsRequestTimeout, @Nullable BiFunction<Throwable, JsonObject, Boolean> exceptionMiddleware) {
        this.queue = new EventQueue(this::sendEventBatch, flushInterval, maxBatchSize);
        this.gson = gson;
        this.endpoint = endpoint;
        this.projectApiKey = projectApiKey;
        this.personalApiKey = personalApiKey;
        this.defaultEventProperties = gson.toJsonTree(defaultEventProperties).getAsJsonObject();
        this.setPropertyIfAbsent(this.defaultEventProperties, "$lib", DEFAULT_LIBRARY_NAME);
        this.setPropertyIfAbsent(this.defaultEventProperties, "$lib_version", DEFAULT_LIBRARY_VERSION);
        this.eventBatchTimeout = eventBatchTimeout;
        if (this.personalApiKey != null) {
            this.featureFlagFetchTimer = new Timer(this::loadRemoteFeatureFlags, featureFlagsPollingInterval);
        } else {
            if (!allowRemoteFeatureFlagEvaluation) {
                throw new IllegalArgumentException("Personal API key is required when remote feature flag evaluation is disabled");
            }
            this.featureFlagFetchTimer = null;
        }
        this.allowRemoteFeatureFlagEvaluation = allowRemoteFeatureFlagEvaluation;
        this.sendFeatureFlagEvents = sendFeatureFlagEvents;
        this.featureFlagsRequestTimeout = featureFlagsRequestTimeout;
        this.exceptionMiddleware = exceptionMiddleware;
    }

    @Override
    public void shutdown(@NotNull Duration timeout) {
        try {
            this.queue.close(timeout);
            if (this.featureFlagFetchTimer != null) {
                this.featureFlagFetchTimer.close();
            }
            this.httpClient.shutdown();
            this.httpClient.awaitTermination(timeout);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    @Override
    public void flush() {
        this.queue.flush();
    }

    @Override
    public void capture(@NotNull String distinctId, @NotNull String event, @NotNull Object properties) {
        JsonObject eventData = new JsonObject();
        eventData.addProperty("uuid", UUID.randomUUID().toString());
        eventData.addProperty("timestamp", Instant.now().toString());
        eventData.addProperty("distinct_id", PostHogNames.nonNullNonEmpty("distinctId", distinctId));
        eventData.addProperty("event", PostHogNames.nonNullNonEmpty("event", event));
        JsonElement localProps = this.gson.toJsonTree(Objects.requireNonNull(properties));
        if (!(localProps instanceof JsonObject)) {
            throw new IllegalArgumentException("Event properties must be a JSON object");
        }
        JsonObject localPropsObject = (JsonObject)localProps;
        JsonObject eventProps = this.defaultEventProperties.deepCopy();
        for (Map.Entry entry : localPropsObject.entrySet()) {
            eventProps.add((String)entry.getKey(), (JsonElement)entry.getValue());
        }
        eventData.add("properties", (JsonElement)eventProps);
        this.queue.enqueue(eventData);
    }

    private void sendEventBatch(@NotNull JsonArray batch) {
        HashMap<String, String> body = new HashMap<String, String>();
        body.put("api_key", this.projectApiKey);
        body.put("batch", (String)batch);
        HttpRequest req = HttpRequest.newBuilder(URI.create(String.format("%s/batch", this.endpoint))).POST(HttpRequest.BodyPublishers.ofString(this.gson.toJson(body))).header("Content-Type", "application/json; charset=utf-8").header("User-Agent", USER_AGENT).timeout(this.eventBatchTimeout).build();
        try {
            HttpResponse<Void> res = this.httpClient.send(req, HttpResponse.BodyHandlers.discarding());
            if (res.statusCode() != 200) {
                throw new RuntimeException(String.format("unexpected response from /batch (%d)", res.statusCode()));
            }
        }
        catch (InterruptedException ignored) {
            Thread.currentThread().interrupt();
        }
        catch (HttpTimeoutException ignored) {
            log.warn("timed out making /batch request");
        }
        catch (Exception e) {
            log.error("failed to make /batch request", (Throwable)e);
        }
    }

    @Override
    @NotNull
    public FeatureFlagState getFeatureFlag(@NotNull String key, @NotNull String distinctId, @Nullable FeatureFlagContext context) {
        boolean sendCalledEvent;
        String featureFlagKey = PostHogNames.nonNullNonEmpty("key", key);
        FeatureFlagContext featureFlagContext = Objects.requireNonNullElse(context, FeatureFlagContext.EMPTY);
        boolean allowRemoteEval = featureFlagContext.allowRemoteEvaluation() != null ? featureFlagContext.allowRemoteEvaluation() : this.allowRemoteFeatureFlagEvaluation;
        FeatureFlagState result = null;
        if (this.featureFlags != null) {
            result = FeatureFlagState.REMOTE_EVAL_NOT_ALLOWED;
            FeatureFlagsResponse.Flag flag = this.featureFlags.get(featureFlagKey);
            if (flag != null) {
                result = FeatureFlagEvaluator.evaluateFeatureFlag(this.gson, flag, distinctId, featureFlagContext);
            }
        }
        if (allowRemoteEval && (result == null || result.isInconclusive())) {
            try {
                JsonObject response = this.decide(distinctId, featureFlagContext);
                JsonObject featureFlags = response.getAsJsonObject("featureFlags");
                JsonObject featureFlagPayloads = response.has("featureFlagPayloads") ? response.getAsJsonObject("featureFlagPayloads") : new JsonObject();
                result = new FeatureFlagState(featureFlags, featureFlagPayloads, featureFlagKey);
            }
            catch (InterruptedException ignored) {
                result = FeatureFlagState.DISABLED;
            }
        }
        boolean bl = sendCalledEvent = featureFlagContext.sendFeatureFlagEvents() != null ? featureFlagContext.sendFeatureFlagEvents() : this.sendFeatureFlagEvents;
        if (sendCalledEvent && this.trackCapturedFeatureFlagCall(distinctId, featureFlagKey)) {
            this.capture(distinctId, "$feature_flag_called", Map.of("$feature_flag", featureFlagKey, "$feature_flag_response", Objects.requireNonNullElse(result.getVariant(), String.valueOf(result.isEnabled())), "$feature_flag_errored", result.isInconclusive()));
        }
        return result;
    }

    @Override
    @NotNull
    public FeatureFlagStates getAllFeatureFlags(@NotNull String distinctId, @Nullable FeatureFlagContext context) {
        FeatureFlagContext featureFlagContext = Objects.requireNonNullElse(context, FeatureFlagContext.EMPTY);
        boolean allowRemoteEval = featureFlagContext.allowRemoteEvaluation() != null ? featureFlagContext.allowRemoteEvaluation() : this.allowRemoteFeatureFlagEvaluation;
        boolean needsLocalEvaluation = true;
        HashMap<String, FeatureFlagState> result = new HashMap<String, FeatureFlagState>();
        if (this.featureFlags != null) {
            needsLocalEvaluation = false;
            for (FeatureFlagsResponse.Flag flag : this.featureFlags.values()) {
                FeatureFlagState state = FeatureFlagEvaluator.evaluateFeatureFlag(this.gson, flag, distinctId, featureFlagContext);
                result.put(flag.key(), state);
                if (!allowRemoteEval || !state.isInconclusive()) continue;
                needsLocalEvaluation = true;
                break;
            }
        }
        if (!allowRemoteEval || !needsLocalEvaluation) {
            return new FeatureFlagStates(result);
        }
        try {
            JsonObject response = this.decide(distinctId, featureFlagContext);
            JsonObject featureFlags = response.getAsJsonObject("featureFlags");
            JsonObject featureFlagsPayloads = response.has("featureFlagPayloads") ? response.getAsJsonObject("featureFlagPayloads") : new JsonObject();
            HashMap<String, FeatureFlagState> states = new HashMap<String, FeatureFlagState>();
            if (featureFlags != null) {
                for (String key : featureFlags.keySet()) {
                    states.put(key, new FeatureFlagState(featureFlags, featureFlagsPayloads, key));
                }
            }
            return new FeatureFlagStates(states);
        }
        catch (InterruptedException ignored) {
            return FeatureFlagStates.EMPTY;
        }
    }

    @Override
    public void reloadFeatureFlags() {
        if (this.featureFlagFetchTimer == null) {
            throw new UnsupportedOperationException("Local feature flag evaluation is not enabled");
        }
        this.featureFlagFetchTimer.wakeup();
    }

    @Blocking
    private void loadRemoteFeatureFlags() {
        if (this.personalApiKey == null) {
            return;
        }
        HttpRequest req = HttpRequest.newBuilder(URI.create(String.format("%s/api/feature_flag/local_evaluation", this.endpoint))).header("Authorization", String.format("Bearer %s", this.personalApiKey)).header("User-Agent", USER_AGENT).timeout(this.featureFlagsRequestTimeout).build();
        try {
            HttpResponse<String> res = this.httpClient.send(req, HttpResponse.BodyHandlers.ofString());
            if (res.statusCode() != 200) {
                log.error("unexpected response from /api/feature_flag/local_evaluation ({}): {}", (Object)res.statusCode(), (Object)res.body());
            }
            FeatureFlagsResponse resBody = (FeatureFlagsResponse)this.gson.fromJson(res.body(), FeatureFlagsResponse.class);
            HashMap<String, FeatureFlagsResponse.Flag> newFeatureFlags = new HashMap<String, FeatureFlagsResponse.Flag>();
            for (FeatureFlagsResponse.Flag flag : resBody.flags()) {
                newFeatureFlags.put(flag.key(), flag);
            }
            this.featureFlags = Map.copyOf(newFeatureFlags);
        }
        catch (InterruptedException res) {
        }
        catch (HttpTimeoutException e) {
            log.warn("timed out making /api/feature_flag/local_evaluation request", (Throwable)e);
        }
        catch (Exception e) {
            log.error("failed to make /api/feature_flag/local_evaluation request", (Throwable)e);
        }
    }

    @NotNull
    private JsonObject decide(@NotNull String distinctId, @NotNull FeatureFlagContext context) throws InterruptedException {
        HashMap<String, Object> body = new HashMap<String, Object>();
        body.put("api_key", this.projectApiKey);
        body.put("distinct_id", PostHogNames.nonNullNonEmpty("distinctId", distinctId));
        if (context.groups() != null) {
            body.put("groups", context.groups());
        }
        if (context.personProperties() != null) {
            body.put("person_properties", context.personProperties());
        }
        if (context.groupProperties() != null) {
            body.put("group_properties", context.groupProperties());
        }
        HttpRequest req = HttpRequest.newBuilder(URI.create(String.format("%s/decide?v=3", this.endpoint))).POST(HttpRequest.BodyPublishers.ofString(this.gson.toJson(body))).header("Content-Type", "application/json; charset=utf-8").header("User-Agent", USER_AGENT).timeout(this.featureFlagsRequestTimeout).build();
        try {
            HttpResponse<String> res = this.httpClient.send(req, HttpResponse.BodyHandlers.ofString());
            if (res.statusCode() != 200) {
                throw new RuntimeException(String.format("unexpected response from /decide (%d): %s", res.statusCode(), res.body()));
            }
            return (JsonObject)this.gson.fromJson(res.body(), JsonObject.class);
        }
        catch (HttpTimeoutException e) {
            log.warn("timed out making /decide request", (Throwable)e);
            return new JsonObject();
        }
        catch (Exception e) {
            log.error("failed to make /decide request", (Throwable)e);
            return new JsonObject();
        }
    }

    private boolean trackCapturedFeatureFlagCall(@NotNull String distinctId, @NotNull String featureFlagKey) {
        String cacheKey;
        if (this.recentlyCapturedFeatureFlags.size() > 50000) {
            this.recentlyCapturedFeatureFlags.clear();
        }
        return this.recentlyCapturedFeatureFlags.put(cacheKey = distinctId + featureFlagKey, "") == null;
    }

    @Override
    public void captureException(@NotNull Throwable throwable, @Nullable String distinctId, @Nullable Object properties) {
        try {
            JsonObject eventProps;
            JsonObject jsonObject = eventProps = properties != null ? this.gson.toJsonTree(properties).getAsJsonObject() : new JsonObject();
            if (distinctId == null) {
                eventProps.addProperty("$process_person_profile", Boolean.valueOf(false));
                distinctId = UUID.randomUUID().toString();
            }
            eventProps.addProperty("$geoip_disable", Boolean.valueOf(true));
            eventProps.addProperty("$exception_type", throwable.getClass().getSimpleName());
            eventProps.addProperty("$exception_message", throwable.getMessage());
            eventProps.add("$exception_list", (JsonElement)this.buildExceptionList(throwable));
            eventProps.addProperty("$exception_personURL", String.format("%s/project/%s/person/%s", this.endpoint, this.projectApiKey, distinctId));
            if (this.exceptionMiddleware != null && !this.exceptionMiddleware.apply(throwable, eventProps).booleanValue()) {
                return;
            }
            this.capture(distinctId, "$exception", eventProps);
        }
        catch (Exception e) {
            log.error("failed to capture exception", (Throwable)e);
        }
    }

    @NotNull
    private JsonArray buildExceptionList(@NotNull Throwable throwable) {
        JsonArray exceptionList = new JsonArray();
        int parentId = -1;
        for (Throwable exc = throwable; exc != null; exc = exc.getCause()) {
            exceptionList.add((JsonElement)this.buildExceptionInterface(exc, parentId));
            ++parentId;
        }
        return exceptionList;
    }

    @NotNull
    private JsonObject buildExceptionInterface(@NotNull Throwable exc, int parentId) {
        JsonObject exception = new JsonObject();
        exception.addProperty("type", exc.getClass().getSimpleName());
        exception.addProperty("module", exc.getClass().getPackageName());
        exception.addProperty("value", Objects.requireNonNullElse(exc.getMessage(), ""));
        JsonObject mechanism = new JsonObject();
        mechanism.addProperty("type", "generic");
        mechanism.addProperty("handled", Boolean.valueOf(true));
        mechanism.addProperty("exception_id", (Number)(parentId + 1));
        if (parentId != -1) {
            mechanism.addProperty("type", "chained");
            mechanism.addProperty("parent_id", (Number)parentId);
        }
        exception.add("mechanism", (JsonElement)mechanism);
        JsonObject stackTrace = new JsonObject();
        stackTrace.add("frames", (JsonElement)this.getStackFrames(exc.getStackTrace()));
        stackTrace.addProperty("type", "raw");
        exception.add("stacktrace", (JsonElement)stackTrace);
        return exception;
    }

    @NotNull
    private JsonArray getStackFrames(StackTraceElement[] elements) {
        int startFrame = Math.max(elements.length - 100, 0);
        JsonArray stackFrames = new JsonArray();
        for (int i = elements.length - 1; i >= startFrame; --i) {
            StackTraceElement element = elements[i];
            if (element == null) continue;
            JsonObject frame = new JsonObject();
            frame.addProperty("platform", "python");
            Object fileName = element.getClassName();
            if (element.getModuleName() != null) {
                fileName = element.getModuleName() + "/" + (String)fileName;
            }
            frame.addProperty("filename", (String)fileName);
            frame.addProperty("abs_path", element.getFileName());
            frame.addProperty("module", element.getClassName());
            frame.addProperty("function", element.getMethodName());
            if (element.getLineNumber() >= 0) {
                frame.addProperty("lineno", (Number)element.getLineNumber());
            }
            frame.add("pre_context", (JsonElement)new JsonArray());
            frame.add("context_line", (JsonElement)JsonNull.INSTANCE);
            frame.add("post_context", (JsonElement)new JsonArray());
            frame.addProperty("in_app", Boolean.valueOf(element.getModuleName() == null || !element.getModuleName().startsWith("java.")));
            stackFrames.add((JsonElement)frame);
        }
        return stackFrames;
    }

    private void setPropertyIfAbsent(@NotNull JsonObject object, @NotNull String key, @NotNull String value) {
        if (!object.has(key)) {
            object.addProperty(key, value);
        }
    }
}

