package io.k8s.api.scheduling.v1

import dev.hnaderi.k8s._
import dev.hnaderi.k8s.utils._

/** PriorityClass defines mapping from a priority class name to the priority integer value. The value can be any valid integer. */
final case class PriorityClass(
  value : Int,
  description : Option[String] = None,
  globalDefault : Option[Boolean] = None,
  preemptionPolicy : Option[String] = None,
  metadata : Option[io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta] = None
) extends KObject {
  protected val _resourceKind = ResourceKind("scheduling.k8s.io", "PriorityClass", "v1")


  /** Returns a new data with value set to new value */
  def withValue(value: Int) : PriorityClass = copy(value = value)
  /** transforms value to result of function */
  def mapValue(f: Int => Int) : PriorityClass = copy(value = f(value))

  /** Returns a new data with description set to new value */
  def withDescription(value: String) : PriorityClass = copy(description = Some(value))
  /** if description has a value, transforms to the result of function*/
  def mapDescription(f: String => String) : PriorityClass = copy(description = description.map(f))

  /** Returns a new data with globalDefault set to new value */
  def withGlobalDefault(value: Boolean) : PriorityClass = copy(globalDefault = Some(value))
  /** if globalDefault has a value, transforms to the result of function*/
  def mapGlobalDefault(f: Boolean => Boolean) : PriorityClass = copy(globalDefault = globalDefault.map(f))

  /** Returns a new data with preemptionPolicy set to new value */
  def withPreemptionPolicy(value: String) : PriorityClass = copy(preemptionPolicy = Some(value))
  /** if preemptionPolicy has a value, transforms to the result of function*/
  def mapPreemptionPolicy(f: String => String) : PriorityClass = copy(preemptionPolicy = preemptionPolicy.map(f))

  /** Returns a new data with metadata set to new value */
  def withMetadata(value: io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta) : PriorityClass = copy(metadata = Some(value))
  /** if metadata has a value, transforms to the result of function*/
  def mapMetadata(f: io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta => io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta) : PriorityClass = copy(metadata = metadata.map(f))

  override def foldTo[T : Builder] : T = PriorityClass.encoder[T].apply(this)
}

object PriorityClass {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.scheduling.v1.PriorityClass, T] = new Encoder[io.k8s.api.scheduling.v1.PriorityClass, T] {
        def apply(o: io.k8s.api.scheduling.v1.PriorityClass) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("value", o.value)
            .write("description", o.description)
            .write("globalDefault", o.globalDefault)
            .write("preemptionPolicy", o.preemptionPolicy)
            .write("metadata", o.metadata)
            .write("kind", o.kind)
            .write("apiVersion", o.apiVersion)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, PriorityClass] = new Decoder[T, PriorityClass] {
      def apply(t: T): Either[String, PriorityClass] = for {
          obj <- ObjectReader(t)
          value <- obj.read[Int]("value")
          description <- obj.readOpt[String]("description")
          globalDefault <- obj.readOpt[Boolean]("globalDefault")
          preemptionPolicy <- obj.readOpt[String]("preemptionPolicy")
          metadata <- obj.readOpt[io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta]("metadata")
      } yield PriorityClass (
          value = value,
          description = description,
          globalDefault = globalDefault,
          preemptionPolicy = preemptionPolicy,
          metadata = metadata
        )
    }
}

