package io.k8s.api.flowcontrol.v1beta2

import dev.hnaderi.k8s.utils._

/** ResourcePolicyRule is a predicate that matches some resource requests, testing the request's verb and the target resource. A ResourcePolicyRule matches a resource request if and only if: (a) at least one member of verbs matches the request, (b) at least one member of apiGroups matches the request, (c) at least one member of resources matches the request, and (d) either (d1) the request does not specify a namespace (i.e., `Namespace==""`) and clusterScope is true or (d2) the request specifies a namespace and least one member of namespaces matches the request's namespace. */
final case class ResourcePolicyRule(
  verbs : Seq[String],
  apiGroups : Seq[String],
  resources : Seq[String],
  clusterScope : Option[Boolean] = None,
  namespaces : Option[Seq[String]] = None
) {

  /** Returns a new data with verbs set to new value */
  def withVerbs(value: Seq[String]) : ResourcePolicyRule = copy(verbs = value)
  /** Appends new values to verbs */
  def addVerbs(newValues: String*) : ResourcePolicyRule = copy(verbs = verbs ++ newValues)
  /** transforms verbs to result of function */
  def mapVerbs(f: Seq[String] => Seq[String]) : ResourcePolicyRule = copy(verbs = f(verbs))

  /** Returns a new data with apiGroups set to new value */
  def withApiGroups(value: Seq[String]) : ResourcePolicyRule = copy(apiGroups = value)
  /** Appends new values to apiGroups */
  def addApiGroups(newValues: String*) : ResourcePolicyRule = copy(apiGroups = apiGroups ++ newValues)
  /** transforms apiGroups to result of function */
  def mapApiGroups(f: Seq[String] => Seq[String]) : ResourcePolicyRule = copy(apiGroups = f(apiGroups))

  /** Returns a new data with resources set to new value */
  def withResources(value: Seq[String]) : ResourcePolicyRule = copy(resources = value)
  /** Appends new values to resources */
  def addResources(newValues: String*) : ResourcePolicyRule = copy(resources = resources ++ newValues)
  /** transforms resources to result of function */
  def mapResources(f: Seq[String] => Seq[String]) : ResourcePolicyRule = copy(resources = f(resources))

  /** Returns a new data with clusterScope set to new value */
  def withClusterScope(value: Boolean) : ResourcePolicyRule = copy(clusterScope = Some(value))
  /** if clusterScope has a value, transforms to the result of function*/
  def mapClusterScope(f: Boolean => Boolean) : ResourcePolicyRule = copy(clusterScope = clusterScope.map(f))

  /** Returns a new data with namespaces set to new value */
  def withNamespaces(value: Seq[String]) : ResourcePolicyRule = copy(namespaces = Some(value))
  /** Appends new values to namespaces */
  def addNamespaces(newValues: String*) : ResourcePolicyRule = copy(namespaces = Some(namespaces.fold(newValues)(_ ++ newValues)))
  /** if namespaces has a value, transforms to the result of function*/
  def mapNamespaces(f: Seq[String] => Seq[String]) : ResourcePolicyRule = copy(namespaces = namespaces.map(f))
}

object ResourcePolicyRule {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.flowcontrol.v1beta2.ResourcePolicyRule, T] = new Encoder[io.k8s.api.flowcontrol.v1beta2.ResourcePolicyRule, T] {
        def apply(o: io.k8s.api.flowcontrol.v1beta2.ResourcePolicyRule) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("verbs", o.verbs)
            .write("apiGroups", o.apiGroups)
            .write("resources", o.resources)
            .write("clusterScope", o.clusterScope)
            .write("namespaces", o.namespaces)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, ResourcePolicyRule] = new Decoder[T, ResourcePolicyRule] {
      def apply(t: T): Either[String, ResourcePolicyRule] = for {
          obj <- ObjectReader(t)
          verbs <- obj.read[Seq[String]]("verbs")
          apiGroups <- obj.read[Seq[String]]("apiGroups")
          resources <- obj.read[Seq[String]]("resources")
          clusterScope <- obj.readOpt[Boolean]("clusterScope")
          namespaces <- obj.readOpt[Seq[String]]("namespaces")
      } yield ResourcePolicyRule (
          verbs = verbs,
          apiGroups = apiGroups,
          resources = resources,
          clusterScope = clusterScope,
          namespaces = namespaces
        )
    }
}

