package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** ServiceAccountTokenProjection represents a projected service account token volume. This projection can be used to insert a service account token into the pods runtime filesystem for use against APIs (Kubernetes API Server or otherwise). */
final case class ServiceAccountTokenProjection(
  path : String,
  audience : Option[String] = None,
  expirationSeconds : Option[Int] = None
) {

  /** Returns a new data with path set to new value */
  def withPath(value: String) : ServiceAccountTokenProjection = copy(path = value)
  /** transforms path to result of function */
  def mapPath(f: String => String) : ServiceAccountTokenProjection = copy(path = f(path))

  /** Returns a new data with audience set to new value */
  def withAudience(value: String) : ServiceAccountTokenProjection = copy(audience = Some(value))
  /** if audience has a value, transforms to the result of function*/
  def mapAudience(f: String => String) : ServiceAccountTokenProjection = copy(audience = audience.map(f))

  /** Returns a new data with expirationSeconds set to new value */
  def withExpirationSeconds(value: Int) : ServiceAccountTokenProjection = copy(expirationSeconds = Some(value))
  /** if expirationSeconds has a value, transforms to the result of function*/
  def mapExpirationSeconds(f: Int => Int) : ServiceAccountTokenProjection = copy(expirationSeconds = expirationSeconds.map(f))
}

object ServiceAccountTokenProjection {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.core.v1.ServiceAccountTokenProjection, T] = new Encoder[io.k8s.api.core.v1.ServiceAccountTokenProjection, T] {
        def apply(o: io.k8s.api.core.v1.ServiceAccountTokenProjection) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("path", o.path)
            .write("audience", o.audience)
            .write("expirationSeconds", o.expirationSeconds)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, ServiceAccountTokenProjection] = new Decoder[T, ServiceAccountTokenProjection] {
      def apply(t: T): Either[String, ServiceAccountTokenProjection] = for {
          obj <- ObjectReader(t)
          path <- obj.read[String]("path")
          audience <- obj.readOpt[String]("audience")
          expirationSeconds <- obj.readOpt[Int]("expirationSeconds")
      } yield ServiceAccountTokenProjection (
          path = path,
          audience = audience,
          expirationSeconds = expirationSeconds
        )
    }
}

