package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** ResourceQuotaSpec defines the desired hard limits to enforce for Quota. */
final case class ResourceQuotaSpec(
  hard : Option[Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]] = None,
  scopeSelector : Option[io.k8s.api.core.v1.ScopeSelector] = None,
  scopes : Option[Seq[String]] = None
) {

  /** Returns a new data with hard set to new value */
  def withHard(value: Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]) : ResourceQuotaSpec = copy(hard = Some(value))
  /** Adds new values to hard */
  def addHard(newValues: (String, io.k8s.apimachinery.pkg.api.resource.Quantity)*) : ResourceQuotaSpec = copy(hard = Some(hard.fold(newValues.toMap)(_ ++ newValues)))
  /** if hard has a value, transforms to the result of function*/
  def mapHard(f: Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity] => Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]) : ResourceQuotaSpec = copy(hard = hard.map(f))

  /** Returns a new data with scopeSelector set to new value */
  def withScopeSelector(value: io.k8s.api.core.v1.ScopeSelector) : ResourceQuotaSpec = copy(scopeSelector = Some(value))
  /** if scopeSelector has a value, transforms to the result of function*/
  def mapScopeSelector(f: io.k8s.api.core.v1.ScopeSelector => io.k8s.api.core.v1.ScopeSelector) : ResourceQuotaSpec = copy(scopeSelector = scopeSelector.map(f))

  /** Returns a new data with scopes set to new value */
  def withScopes(value: Seq[String]) : ResourceQuotaSpec = copy(scopes = Some(value))
  /** Appends new values to scopes */
  def addScopes(newValues: String*) : ResourceQuotaSpec = copy(scopes = Some(scopes.fold(newValues)(_ ++ newValues)))
  /** if scopes has a value, transforms to the result of function*/
  def mapScopes(f: Seq[String] => Seq[String]) : ResourceQuotaSpec = copy(scopes = scopes.map(f))
}

object ResourceQuotaSpec {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.core.v1.ResourceQuotaSpec, T] = new Encoder[io.k8s.api.core.v1.ResourceQuotaSpec, T] {
        def apply(o: io.k8s.api.core.v1.ResourceQuotaSpec) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("hard", o.hard)
            .write("scopeSelector", o.scopeSelector)
            .write("scopes", o.scopes)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, ResourceQuotaSpec] = new Decoder[T, ResourceQuotaSpec] {
      def apply(t: T): Either[String, ResourceQuotaSpec] = for {
          obj <- ObjectReader(t)
          hard <- obj.readOpt[Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]]("hard")
          scopeSelector <- obj.readOpt[io.k8s.api.core.v1.ScopeSelector]("scopeSelector")
          scopes <- obj.readOpt[Seq[String]]("scopes")
      } yield ResourceQuotaSpec (
          hard = hard,
          scopeSelector = scopeSelector,
          scopes = scopes
        )
    }
}

