package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** ISCSIPersistentVolumeSource represents an ISCSI disk. ISCSI volumes can only be mounted as read/write once. ISCSI volumes support ownership management and SELinux relabeling. */
final case class ISCSIPersistentVolumeSource(
  iqn : String,
  targetPortal : String,
  lun : Int,
  portals : Option[Seq[String]] = None,
  readOnly : Option[Boolean] = None,
  chapAuthDiscovery : Option[Boolean] = None,
  fsType : Option[String] = None,
  secretRef : Option[io.k8s.api.core.v1.SecretReference] = None,
  initiatorName : Option[String] = None,
  iscsiInterface : Option[String] = None,
  chapAuthSession : Option[Boolean] = None
) {

  /** Returns a new data with iqn set to new value */
  def withIqn(value: String) : ISCSIPersistentVolumeSource = copy(iqn = value)
  /** transforms iqn to result of function */
  def mapIqn(f: String => String) : ISCSIPersistentVolumeSource = copy(iqn = f(iqn))

  /** Returns a new data with targetPortal set to new value */
  def withTargetPortal(value: String) : ISCSIPersistentVolumeSource = copy(targetPortal = value)
  /** transforms targetPortal to result of function */
  def mapTargetPortal(f: String => String) : ISCSIPersistentVolumeSource = copy(targetPortal = f(targetPortal))

  /** Returns a new data with lun set to new value */
  def withLun(value: Int) : ISCSIPersistentVolumeSource = copy(lun = value)
  /** transforms lun to result of function */
  def mapLun(f: Int => Int) : ISCSIPersistentVolumeSource = copy(lun = f(lun))

  /** Returns a new data with portals set to new value */
  def withPortals(value: Seq[String]) : ISCSIPersistentVolumeSource = copy(portals = Some(value))
  /** Appends new values to portals */
  def addPortals(newValues: String*) : ISCSIPersistentVolumeSource = copy(portals = Some(portals.fold(newValues)(_ ++ newValues)))
  /** if portals has a value, transforms to the result of function*/
  def mapPortals(f: Seq[String] => Seq[String]) : ISCSIPersistentVolumeSource = copy(portals = portals.map(f))

  /** Returns a new data with readOnly set to new value */
  def withReadOnly(value: Boolean) : ISCSIPersistentVolumeSource = copy(readOnly = Some(value))
  /** if readOnly has a value, transforms to the result of function*/
  def mapReadOnly(f: Boolean => Boolean) : ISCSIPersistentVolumeSource = copy(readOnly = readOnly.map(f))

  /** Returns a new data with chapAuthDiscovery set to new value */
  def withChapAuthDiscovery(value: Boolean) : ISCSIPersistentVolumeSource = copy(chapAuthDiscovery = Some(value))
  /** if chapAuthDiscovery has a value, transforms to the result of function*/
  def mapChapAuthDiscovery(f: Boolean => Boolean) : ISCSIPersistentVolumeSource = copy(chapAuthDiscovery = chapAuthDiscovery.map(f))

  /** Returns a new data with fsType set to new value */
  def withFsType(value: String) : ISCSIPersistentVolumeSource = copy(fsType = Some(value))
  /** if fsType has a value, transforms to the result of function*/
  def mapFsType(f: String => String) : ISCSIPersistentVolumeSource = copy(fsType = fsType.map(f))

  /** Returns a new data with secretRef set to new value */
  def withSecretRef(value: io.k8s.api.core.v1.SecretReference) : ISCSIPersistentVolumeSource = copy(secretRef = Some(value))
  /** if secretRef has a value, transforms to the result of function*/
  def mapSecretRef(f: io.k8s.api.core.v1.SecretReference => io.k8s.api.core.v1.SecretReference) : ISCSIPersistentVolumeSource = copy(secretRef = secretRef.map(f))

  /** Returns a new data with initiatorName set to new value */
  def withInitiatorName(value: String) : ISCSIPersistentVolumeSource = copy(initiatorName = Some(value))
  /** if initiatorName has a value, transforms to the result of function*/
  def mapInitiatorName(f: String => String) : ISCSIPersistentVolumeSource = copy(initiatorName = initiatorName.map(f))

  /** Returns a new data with iscsiInterface set to new value */
  def withIscsiInterface(value: String) : ISCSIPersistentVolumeSource = copy(iscsiInterface = Some(value))
  /** if iscsiInterface has a value, transforms to the result of function*/
  def mapIscsiInterface(f: String => String) : ISCSIPersistentVolumeSource = copy(iscsiInterface = iscsiInterface.map(f))

  /** Returns a new data with chapAuthSession set to new value */
  def withChapAuthSession(value: Boolean) : ISCSIPersistentVolumeSource = copy(chapAuthSession = Some(value))
  /** if chapAuthSession has a value, transforms to the result of function*/
  def mapChapAuthSession(f: Boolean => Boolean) : ISCSIPersistentVolumeSource = copy(chapAuthSession = chapAuthSession.map(f))
}

object ISCSIPersistentVolumeSource {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.core.v1.ISCSIPersistentVolumeSource, T] = new Encoder[io.k8s.api.core.v1.ISCSIPersistentVolumeSource, T] {
        def apply(o: io.k8s.api.core.v1.ISCSIPersistentVolumeSource) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("iqn", o.iqn)
            .write("targetPortal", o.targetPortal)
            .write("lun", o.lun)
            .write("portals", o.portals)
            .write("readOnly", o.readOnly)
            .write("chapAuthDiscovery", o.chapAuthDiscovery)
            .write("fsType", o.fsType)
            .write("secretRef", o.secretRef)
            .write("initiatorName", o.initiatorName)
            .write("iscsiInterface", o.iscsiInterface)
            .write("chapAuthSession", o.chapAuthSession)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, ISCSIPersistentVolumeSource] = new Decoder[T, ISCSIPersistentVolumeSource] {
      def apply(t: T): Either[String, ISCSIPersistentVolumeSource] = for {
          obj <- ObjectReader(t)
          iqn <- obj.read[String]("iqn")
          targetPortal <- obj.read[String]("targetPortal")
          lun <- obj.read[Int]("lun")
          portals <- obj.readOpt[Seq[String]]("portals")
          readOnly <- obj.readOpt[Boolean]("readOnly")
          chapAuthDiscovery <- obj.readOpt[Boolean]("chapAuthDiscovery")
          fsType <- obj.readOpt[String]("fsType")
          secretRef <- obj.readOpt[io.k8s.api.core.v1.SecretReference]("secretRef")
          initiatorName <- obj.readOpt[String]("initiatorName")
          iscsiInterface <- obj.readOpt[String]("iscsiInterface")
          chapAuthSession <- obj.readOpt[Boolean]("chapAuthSession")
      } yield ISCSIPersistentVolumeSource (
          iqn = iqn,
          targetPortal = targetPortal,
          lun = lun,
          portals = portals,
          readOnly = readOnly,
          chapAuthDiscovery = chapAuthDiscovery,
          fsType = fsType,
          secretRef = secretRef,
          initiatorName = initiatorName,
          iscsiInterface = iscsiInterface,
          chapAuthSession = chapAuthSession
        )
    }
}

