package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** FlexPersistentVolumeSource represents a generic persistent volume resource that is provisioned/attached using an exec based plugin. */
final case class FlexPersistentVolumeSource(
  driver : String,
  readOnly : Option[Boolean] = None,
  fsType : Option[String] = None,
  secretRef : Option[io.k8s.api.core.v1.SecretReference] = None,
  options : Option[Map[String, String]] = None
) {

  /** Returns a new data with driver set to new value */
  def withDriver(value: String) : FlexPersistentVolumeSource = copy(driver = value)
  /** transforms driver to result of function */
  def mapDriver(f: String => String) : FlexPersistentVolumeSource = copy(driver = f(driver))

  /** Returns a new data with readOnly set to new value */
  def withReadOnly(value: Boolean) : FlexPersistentVolumeSource = copy(readOnly = Some(value))
  /** if readOnly has a value, transforms to the result of function*/
  def mapReadOnly(f: Boolean => Boolean) : FlexPersistentVolumeSource = copy(readOnly = readOnly.map(f))

  /** Returns a new data with fsType set to new value */
  def withFsType(value: String) : FlexPersistentVolumeSource = copy(fsType = Some(value))
  /** if fsType has a value, transforms to the result of function*/
  def mapFsType(f: String => String) : FlexPersistentVolumeSource = copy(fsType = fsType.map(f))

  /** Returns a new data with secretRef set to new value */
  def withSecretRef(value: io.k8s.api.core.v1.SecretReference) : FlexPersistentVolumeSource = copy(secretRef = Some(value))
  /** if secretRef has a value, transforms to the result of function*/
  def mapSecretRef(f: io.k8s.api.core.v1.SecretReference => io.k8s.api.core.v1.SecretReference) : FlexPersistentVolumeSource = copy(secretRef = secretRef.map(f))

  /** Returns a new data with options set to new value */
  def withOptions(value: Map[String, String]) : FlexPersistentVolumeSource = copy(options = Some(value))
  /** Adds new values to options */
  def addOptions(newValues: (String, String)*) : FlexPersistentVolumeSource = copy(options = Some(options.fold(newValues.toMap)(_ ++ newValues)))
  /** if options has a value, transforms to the result of function*/
  def mapOptions(f: Map[String, String] => Map[String, String]) : FlexPersistentVolumeSource = copy(options = options.map(f))
}

object FlexPersistentVolumeSource {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.core.v1.FlexPersistentVolumeSource, T] = new Encoder[io.k8s.api.core.v1.FlexPersistentVolumeSource, T] {
        def apply(o: io.k8s.api.core.v1.FlexPersistentVolumeSource) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("driver", o.driver)
            .write("readOnly", o.readOnly)
            .write("fsType", o.fsType)
            .write("secretRef", o.secretRef)
            .write("options", o.options)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, FlexPersistentVolumeSource] = new Decoder[T, FlexPersistentVolumeSource] {
      def apply(t: T): Either[String, FlexPersistentVolumeSource] = for {
          obj <- ObjectReader(t)
          driver <- obj.read[String]("driver")
          readOnly <- obj.readOpt[Boolean]("readOnly")
          fsType <- obj.readOpt[String]("fsType")
          secretRef <- obj.readOpt[io.k8s.api.core.v1.SecretReference]("secretRef")
          options <- obj.readOpt[Map[String, String]]("options")
      } yield FlexPersistentVolumeSource (
          driver = driver,
          readOnly = readOnly,
          fsType = fsType,
          secretRef = secretRef,
          options = options
        )
    }
}

