package io.k8s.api.core.v1

import dev.hnaderi.k8s._
import dev.hnaderi.k8s.utils._

/** Endpoints is a collection of endpoints that implement the actual service. Example:

	 Name: "mysvc",
	 Subsets: [
	   {
	     Addresses: [{"ip": "10.10.1.1"}, {"ip": "10.10.2.2"}],
	     Ports: [{"name": "a", "port": 8675}, {"name": "b", "port": 309}]
	   },
	   {
	     Addresses: [{"ip": "10.10.3.3"}],
	     Ports: [{"name": "a", "port": 93}, {"name": "b", "port": 76}]
	   },
	] */
final case class Endpoints(
  metadata : Option[io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta] = None,
  subsets : Option[Seq[io.k8s.api.core.v1.EndpointSubset]] = None
) extends KObject {
  protected val _resourceKind = ResourceKind("", "Endpoints", "v1")


  /** Returns a new data with metadata set to new value */
  def withMetadata(value: io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta) : Endpoints = copy(metadata = Some(value))
  /** if metadata has a value, transforms to the result of function*/
  def mapMetadata(f: io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta => io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta) : Endpoints = copy(metadata = metadata.map(f))

  /** Returns a new data with subsets set to new value */
  def withSubsets(value: Seq[io.k8s.api.core.v1.EndpointSubset]) : Endpoints = copy(subsets = Some(value))
  /** Appends new values to subsets */
  def addSubsets(newValues: io.k8s.api.core.v1.EndpointSubset*) : Endpoints = copy(subsets = Some(subsets.fold(newValues)(_ ++ newValues)))
  /** if subsets has a value, transforms to the result of function*/
  def mapSubsets(f: Seq[io.k8s.api.core.v1.EndpointSubset] => Seq[io.k8s.api.core.v1.EndpointSubset]) : Endpoints = copy(subsets = subsets.map(f))

  override def foldTo[T : Builder] : T = Endpoints.encoder[T].apply(this)
}

object Endpoints {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.core.v1.Endpoints, T] = new Encoder[io.k8s.api.core.v1.Endpoints, T] {
        def apply(o: io.k8s.api.core.v1.Endpoints) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("metadata", o.metadata)
            .write("subsets", o.subsets)
            .write("kind", o.kind)
            .write("apiVersion", o.apiVersion)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, Endpoints] = new Decoder[T, Endpoints] {
      def apply(t: T): Either[String, Endpoints] = for {
          obj <- ObjectReader(t)
          metadata <- obj.readOpt[io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta]("metadata")
          subsets <- obj.readOpt[Seq[io.k8s.api.core.v1.EndpointSubset]]("subsets")
      } yield Endpoints (
          metadata = metadata,
          subsets = subsets
        )
    }
}

