package io.k8s.api.authorization.v1

import dev.hnaderi.k8s.utils._

/** NonResourceRule holds information that describes a rule for the non-resource */
final case class NonResourceRule(
  verbs : Seq[String],
  nonResourceURLs : Option[Seq[String]] = None
) {

  /** Returns a new data with verbs set to new value */
  def withVerbs(value: Seq[String]) : NonResourceRule = copy(verbs = value)
  /** Appends new values to verbs */
  def addVerbs(newValues: String*) : NonResourceRule = copy(verbs = verbs ++ newValues)
  /** transforms verbs to result of function */
  def mapVerbs(f: Seq[String] => Seq[String]) : NonResourceRule = copy(verbs = f(verbs))

  /** Returns a new data with nonResourceURLs set to new value */
  def withNonResourceURLs(value: Seq[String]) : NonResourceRule = copy(nonResourceURLs = Some(value))
  /** Appends new values to nonResourceURLs */
  def addNonResourceURLs(newValues: String*) : NonResourceRule = copy(nonResourceURLs = Some(nonResourceURLs.fold(newValues)(_ ++ newValues)))
  /** if nonResourceURLs has a value, transforms to the result of function*/
  def mapNonResourceURLs(f: Seq[String] => Seq[String]) : NonResourceRule = copy(nonResourceURLs = nonResourceURLs.map(f))
}

object NonResourceRule {

    implicit def encoder[T](implicit builder : Builder[T]) : Encoder[io.k8s.api.authorization.v1.NonResourceRule, T] = new Encoder[io.k8s.api.authorization.v1.NonResourceRule, T] {
        def apply(o: io.k8s.api.authorization.v1.NonResourceRule) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("verbs", o.verbs)
            .write("nonResourceURLs", o.nonResourceURLs)
            .build
        }
    }

    implicit def decoderOf[T : Reader] : Decoder[T, NonResourceRule] = new Decoder[T, NonResourceRule] {
      def apply(t: T): Either[String, NonResourceRule] = for {
          obj <- ObjectReader(t)
          verbs <- obj.read[Seq[String]]("verbs")
          nonResourceURLs <- obj.readOpt[Seq[String]]("nonResourceURLs")
      } yield NonResourceRule (
          verbs = verbs,
          nonResourceURLs = nonResourceURLs
        )
    }
}

