package io.k8s.apimachinery.pkg.apis.meta.v1

import dev.hnaderi.k8s.utils._

/** FieldSelectorRequirement is a selector that contains values, a key, and an operator that relates the key and values. */
final case class FieldSelectorRequirement(
  key : String,
  operator : String,
  values : Option[Seq[String]] = None
) {

  /** Returns a new data with key set to new value */
  def withKey(value: String) : FieldSelectorRequirement = copy(key = value)
  /** transforms key to result of function */
  def mapKey(f: String => String) : FieldSelectorRequirement = copy(key = f(key))

  /** Returns a new data with operator set to new value */
  def withOperator(value: String) : FieldSelectorRequirement = copy(operator = value)
  /** transforms operator to result of function */
  def mapOperator(f: String => String) : FieldSelectorRequirement = copy(operator = f(operator))

  /** Returns a new data with values set to new value */
  def withValues(value: Seq[String]) : FieldSelectorRequirement = copy(values = Some(value))
  /** Appends new values to values */
  def addValues(newValues: String*) : FieldSelectorRequirement = copy(values = Some(values.fold(newValues)(_ ++ newValues)))
  /** if values has a value, transforms to the result of function*/
  def mapValues(f: Seq[String] => Seq[String]) : FieldSelectorRequirement = copy(values = values.map(f))
}

object FieldSelectorRequirement {

    implicit val encoder : Encoder[io.k8s.apimachinery.pkg.apis.meta.v1.FieldSelectorRequirement] = new Encoder[io.k8s.apimachinery.pkg.apis.meta.v1.FieldSelectorRequirement] {
        def apply[T : Builder](o: io.k8s.apimachinery.pkg.apis.meta.v1.FieldSelectorRequirement) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("key", o.key)
            .write("operator", o.operator)
            .write("values", o.values)
            .build
        }
    }

    implicit val decoder: Decoder[FieldSelectorRequirement] = new Decoder[FieldSelectorRequirement] {
      def apply[T : Reader](t: T): Either[String, FieldSelectorRequirement] = for {
          obj <- ObjectReader(t)
          key <- obj.read[String]("key")
          operator <- obj.read[String]("operator")
          values <- obj.readOpt[Seq[String]]("values")
      } yield FieldSelectorRequirement (
          key = key,
          operator = operator,
          values = values
        )
    }
}

