package io.k8s.api.resource.v1alpha3

import dev.hnaderi.k8s.utils._

/** ResourceSliceSpec contains the information published by the driver in one ResourceSlice. */
final case class ResourceSliceSpec(
  driver : String,
  pool : io.k8s.api.resource.v1alpha3.ResourcePool,
  nodeName : Option[String] = None,
  nodeSelector : Option[io.k8s.api.core.v1.NodeSelector] = None,
  allNodes : Option[Boolean] = None,
  devices : Option[Seq[io.k8s.api.resource.v1alpha3.Device]] = None
) {

  /** Returns a new data with driver set to new value */
  def withDriver(value: String) : ResourceSliceSpec = copy(driver = value)
  /** transforms driver to result of function */
  def mapDriver(f: String => String) : ResourceSliceSpec = copy(driver = f(driver))

  /** Returns a new data with pool set to new value */
  def withPool(value: io.k8s.api.resource.v1alpha3.ResourcePool) : ResourceSliceSpec = copy(pool = value)
  /** transforms pool to result of function */
  def mapPool(f: io.k8s.api.resource.v1alpha3.ResourcePool => io.k8s.api.resource.v1alpha3.ResourcePool) : ResourceSliceSpec = copy(pool = f(pool))

  /** Returns a new data with nodeName set to new value */
  def withNodeName(value: String) : ResourceSliceSpec = copy(nodeName = Some(value))
  /** if nodeName has a value, transforms to the result of function*/
  def mapNodeName(f: String => String) : ResourceSliceSpec = copy(nodeName = nodeName.map(f))

  /** Returns a new data with nodeSelector set to new value */
  def withNodeSelector(value: io.k8s.api.core.v1.NodeSelector) : ResourceSliceSpec = copy(nodeSelector = Some(value))
  /** if nodeSelector has a value, transforms to the result of function*/
  def mapNodeSelector(f: io.k8s.api.core.v1.NodeSelector => io.k8s.api.core.v1.NodeSelector) : ResourceSliceSpec = copy(nodeSelector = nodeSelector.map(f))

  /** Returns a new data with allNodes set to new value */
  def withAllNodes(value: Boolean) : ResourceSliceSpec = copy(allNodes = Some(value))
  /** if allNodes has a value, transforms to the result of function*/
  def mapAllNodes(f: Boolean => Boolean) : ResourceSliceSpec = copy(allNodes = allNodes.map(f))

  /** Returns a new data with devices set to new value */
  def withDevices(value: Seq[io.k8s.api.resource.v1alpha3.Device]) : ResourceSliceSpec = copy(devices = Some(value))
  /** Appends new values to devices */
  def addDevices(newValues: io.k8s.api.resource.v1alpha3.Device*) : ResourceSliceSpec = copy(devices = Some(devices.fold(newValues)(_ ++ newValues)))
  /** if devices has a value, transforms to the result of function*/
  def mapDevices(f: Seq[io.k8s.api.resource.v1alpha3.Device] => Seq[io.k8s.api.resource.v1alpha3.Device]) : ResourceSliceSpec = copy(devices = devices.map(f))
}

object ResourceSliceSpec {

    implicit val encoder : Encoder[io.k8s.api.resource.v1alpha3.ResourceSliceSpec] = new Encoder[io.k8s.api.resource.v1alpha3.ResourceSliceSpec] {
        def apply[T : Builder](o: io.k8s.api.resource.v1alpha3.ResourceSliceSpec) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("driver", o.driver)
            .write("pool", o.pool)
            .write("nodeName", o.nodeName)
            .write("nodeSelector", o.nodeSelector)
            .write("allNodes", o.allNodes)
            .write("devices", o.devices)
            .build
        }
    }

    implicit val decoder: Decoder[ResourceSliceSpec] = new Decoder[ResourceSliceSpec] {
      def apply[T : Reader](t: T): Either[String, ResourceSliceSpec] = for {
          obj <- ObjectReader(t)
          driver <- obj.read[String]("driver")
          pool <- obj.read[io.k8s.api.resource.v1alpha3.ResourcePool]("pool")
          nodeName <- obj.readOpt[String]("nodeName")
          nodeSelector <- obj.readOpt[io.k8s.api.core.v1.NodeSelector]("nodeSelector")
          allNodes <- obj.readOpt[Boolean]("allNodes")
          devices <- obj.readOpt[Seq[io.k8s.api.resource.v1alpha3.Device]]("devices")
      } yield ResourceSliceSpec (
          driver = driver,
          pool = pool,
          nodeName = nodeName,
          nodeSelector = nodeSelector,
          allNodes = allNodes,
          devices = devices
        )
    }
}

