package io.k8s.api.resource.v1alpha3

import dev.hnaderi.k8s.utils._

/** DeviceRequest is a request for devices required for a claim. This is typically a request for a single resource like a device, but can also ask for several identical devices.

A DeviceClassName is currently required. Clients must check that it is indeed set. It's absence indicates that something changed in a way that is not supported by the client yet, in which case it must refuse to handle the request. */
final case class DeviceRequest(
  name : String,
  deviceClassName : String,
  count : Option[Long] = None,
  adminAccess : Option[Boolean] = None,
  selectors : Option[Seq[io.k8s.api.resource.v1alpha3.DeviceSelector]] = None,
  allocationMode : Option[String] = None
) {

  /** Returns a new data with name set to new value */
  def withName(value: String) : DeviceRequest = copy(name = value)
  /** transforms name to result of function */
  def mapName(f: String => String) : DeviceRequest = copy(name = f(name))

  /** Returns a new data with deviceClassName set to new value */
  def withDeviceClassName(value: String) : DeviceRequest = copy(deviceClassName = value)
  /** transforms deviceClassName to result of function */
  def mapDeviceClassName(f: String => String) : DeviceRequest = copy(deviceClassName = f(deviceClassName))

  /** Returns a new data with count set to new value */
  def withCount(value: Long) : DeviceRequest = copy(count = Some(value))
  /** if count has a value, transforms to the result of function*/
  def mapCount(f: Long => Long) : DeviceRequest = copy(count = count.map(f))

  /** Returns a new data with adminAccess set to new value */
  def withAdminAccess(value: Boolean) : DeviceRequest = copy(adminAccess = Some(value))
  /** if adminAccess has a value, transforms to the result of function*/
  def mapAdminAccess(f: Boolean => Boolean) : DeviceRequest = copy(adminAccess = adminAccess.map(f))

  /** Returns a new data with selectors set to new value */
  def withSelectors(value: Seq[io.k8s.api.resource.v1alpha3.DeviceSelector]) : DeviceRequest = copy(selectors = Some(value))
  /** Appends new values to selectors */
  def addSelectors(newValues: io.k8s.api.resource.v1alpha3.DeviceSelector*) : DeviceRequest = copy(selectors = Some(selectors.fold(newValues)(_ ++ newValues)))
  /** if selectors has a value, transforms to the result of function*/
  def mapSelectors(f: Seq[io.k8s.api.resource.v1alpha3.DeviceSelector] => Seq[io.k8s.api.resource.v1alpha3.DeviceSelector]) : DeviceRequest = copy(selectors = selectors.map(f))

  /** Returns a new data with allocationMode set to new value */
  def withAllocationMode(value: String) : DeviceRequest = copy(allocationMode = Some(value))
  /** if allocationMode has a value, transforms to the result of function*/
  def mapAllocationMode(f: String => String) : DeviceRequest = copy(allocationMode = allocationMode.map(f))
}

object DeviceRequest {

    implicit val encoder : Encoder[io.k8s.api.resource.v1alpha3.DeviceRequest] = new Encoder[io.k8s.api.resource.v1alpha3.DeviceRequest] {
        def apply[T : Builder](o: io.k8s.api.resource.v1alpha3.DeviceRequest) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("name", o.name)
            .write("deviceClassName", o.deviceClassName)
            .write("count", o.count)
            .write("adminAccess", o.adminAccess)
            .write("selectors", o.selectors)
            .write("allocationMode", o.allocationMode)
            .build
        }
    }

    implicit val decoder: Decoder[DeviceRequest] = new Decoder[DeviceRequest] {
      def apply[T : Reader](t: T): Either[String, DeviceRequest] = for {
          obj <- ObjectReader(t)
          name <- obj.read[String]("name")
          deviceClassName <- obj.read[String]("deviceClassName")
          count <- obj.readOpt[Long]("count")
          adminAccess <- obj.readOpt[Boolean]("adminAccess")
          selectors <- obj.readOpt[Seq[io.k8s.api.resource.v1alpha3.DeviceSelector]]("selectors")
          allocationMode <- obj.readOpt[String]("allocationMode")
      } yield DeviceRequest (
          name = name,
          deviceClassName = deviceClassName,
          count = count,
          adminAccess = adminAccess,
          selectors = selectors,
          allocationMode = allocationMode
        )
    }
}

