package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** ResourceHealth represents the health of a resource. It has the latest device health information. This is a part of KEP https://kep.k8s.io/4680 and historical health changes are planned to be added in future iterations of a KEP. */
final case class ResourceHealth(
  resourceID : String,
  health : Option[String] = None
) {

  /** Returns a new data with resourceID set to new value */
  def withResourceID(value: String) : ResourceHealth = copy(resourceID = value)
  /** transforms resourceID to result of function */
  def mapResourceID(f: String => String) : ResourceHealth = copy(resourceID = f(resourceID))

  /** Returns a new data with health set to new value */
  def withHealth(value: String) : ResourceHealth = copy(health = Some(value))
  /** if health has a value, transforms to the result of function*/
  def mapHealth(f: String => String) : ResourceHealth = copy(health = health.map(f))
}

object ResourceHealth {

    implicit val encoder : Encoder[io.k8s.api.core.v1.ResourceHealth] = new Encoder[io.k8s.api.core.v1.ResourceHealth] {
        def apply[T : Builder](o: io.k8s.api.core.v1.ResourceHealth) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("resourceID", o.resourceID)
            .write("health", o.health)
            .build
        }
    }

    implicit val decoder: Decoder[ResourceHealth] = new Decoder[ResourceHealth] {
      def apply[T : Reader](t: T): Either[String, ResourceHealth] = for {
          obj <- ObjectReader(t)
          resourceID <- obj.read[String]("resourceID")
          health <- obj.readOpt[String]("health")
      } yield ResourceHealth (
          resourceID = resourceID,
          health = health
        )
    }
}

