package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** PodResourceClaim references exactly one ResourceClaim, either directly or by naming a ResourceClaimTemplate which is then turned into a ResourceClaim for the pod.

It adds a name to it that uniquely identifies the ResourceClaim inside the Pod. Containers that need access to the ResourceClaim reference it with this name. */
final case class PodResourceClaim(
  name : String,
  resourceClaimName : Option[String] = None,
  resourceClaimTemplateName : Option[String] = None
) {

  /** Returns a new data with name set to new value */
  def withName(value: String) : PodResourceClaim = copy(name = value)
  /** transforms name to result of function */
  def mapName(f: String => String) : PodResourceClaim = copy(name = f(name))

  /** Returns a new data with resourceClaimName set to new value */
  def withResourceClaimName(value: String) : PodResourceClaim = copy(resourceClaimName = Some(value))
  /** if resourceClaimName has a value, transforms to the result of function*/
  def mapResourceClaimName(f: String => String) : PodResourceClaim = copy(resourceClaimName = resourceClaimName.map(f))

  /** Returns a new data with resourceClaimTemplateName set to new value */
  def withResourceClaimTemplateName(value: String) : PodResourceClaim = copy(resourceClaimTemplateName = Some(value))
  /** if resourceClaimTemplateName has a value, transforms to the result of function*/
  def mapResourceClaimTemplateName(f: String => String) : PodResourceClaim = copy(resourceClaimTemplateName = resourceClaimTemplateName.map(f))
}

object PodResourceClaim {

    implicit val encoder : Encoder[io.k8s.api.core.v1.PodResourceClaim] = new Encoder[io.k8s.api.core.v1.PodResourceClaim] {
        def apply[T : Builder](o: io.k8s.api.core.v1.PodResourceClaim) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("name", o.name)
            .write("resourceClaimName", o.resourceClaimName)
            .write("resourceClaimTemplateName", o.resourceClaimTemplateName)
            .build
        }
    }

    implicit val decoder: Decoder[PodResourceClaim] = new Decoder[PodResourceClaim] {
      def apply[T : Reader](t: T): Either[String, PodResourceClaim] = for {
          obj <- ObjectReader(t)
          name <- obj.read[String]("name")
          resourceClaimName <- obj.readOpt[String]("resourceClaimName")
          resourceClaimTemplateName <- obj.readOpt[String]("resourceClaimTemplateName")
      } yield PodResourceClaim (
          name = name,
          resourceClaimName = resourceClaimName,
          resourceClaimTemplateName = resourceClaimTemplateName
        )
    }
}

