package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** LinuxContainerUser represents user identity information in Linux containers */
final case class LinuxContainerUser(
  gid : Long,
  uid : Long,
  supplementalGroups : Option[Seq[Long]] = None
) {

  /** Returns a new data with gid set to new value */
  def withGid(value: Long) : LinuxContainerUser = copy(gid = value)
  /** transforms gid to result of function */
  def mapGid(f: Long => Long) : LinuxContainerUser = copy(gid = f(gid))

  /** Returns a new data with uid set to new value */
  def withUid(value: Long) : LinuxContainerUser = copy(uid = value)
  /** transforms uid to result of function */
  def mapUid(f: Long => Long) : LinuxContainerUser = copy(uid = f(uid))

  /** Returns a new data with supplementalGroups set to new value */
  def withSupplementalGroups(value: Seq[Long]) : LinuxContainerUser = copy(supplementalGroups = Some(value))
  /** Appends new values to supplementalGroups */
  def addSupplementalGroups(newValues: Long*) : LinuxContainerUser = copy(supplementalGroups = Some(supplementalGroups.fold(newValues)(_ ++ newValues)))
  /** if supplementalGroups has a value, transforms to the result of function*/
  def mapSupplementalGroups(f: Seq[Long] => Seq[Long]) : LinuxContainerUser = copy(supplementalGroups = supplementalGroups.map(f))
}

object LinuxContainerUser {

    implicit val encoder : Encoder[io.k8s.api.core.v1.LinuxContainerUser] = new Encoder[io.k8s.api.core.v1.LinuxContainerUser] {
        def apply[T : Builder](o: io.k8s.api.core.v1.LinuxContainerUser) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("gid", o.gid)
            .write("uid", o.uid)
            .write("supplementalGroups", o.supplementalGroups)
            .build
        }
    }

    implicit val decoder: Decoder[LinuxContainerUser] = new Decoder[LinuxContainerUser] {
      def apply[T : Reader](t: T): Either[String, LinuxContainerUser] = for {
          obj <- ObjectReader(t)
          gid <- obj.read[Long]("gid")
          uid <- obj.read[Long]("uid")
          supplementalGroups <- obj.readOpt[Seq[Long]]("supplementalGroups")
      } yield LinuxContainerUser (
          gid = gid,
          uid = uid,
          supplementalGroups = supplementalGroups
        )
    }
}

