package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** ContainerStatus contains details for the current status of this container. */
final case class ContainerStatus(
  name : String,
  image : String,
  restartCount : Int,
  ready : Boolean,
  imageID : String,
  state : Option[io.k8s.api.core.v1.ContainerState] = None,
  allocatedResourcesStatus : Option[Seq[io.k8s.api.core.v1.ResourceStatus]] = None,
  lastState : Option[io.k8s.api.core.v1.ContainerState] = None,
  resources : Option[io.k8s.api.core.v1.ResourceRequirements] = None,
  containerID : Option[String] = None,
  started : Option[Boolean] = None,
  user : Option[io.k8s.api.core.v1.ContainerUser] = None,
  allocatedResources : Option[Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]] = None,
  volumeMounts : Option[Seq[io.k8s.api.core.v1.VolumeMountStatus]] = None
) {

  /** Returns a new data with name set to new value */
  def withName(value: String) : ContainerStatus = copy(name = value)
  /** transforms name to result of function */
  def mapName(f: String => String) : ContainerStatus = copy(name = f(name))

  /** Returns a new data with image set to new value */
  def withImage(value: String) : ContainerStatus = copy(image = value)
  /** transforms image to result of function */
  def mapImage(f: String => String) : ContainerStatus = copy(image = f(image))

  /** Returns a new data with restartCount set to new value */
  def withRestartCount(value: Int) : ContainerStatus = copy(restartCount = value)
  /** transforms restartCount to result of function */
  def mapRestartCount(f: Int => Int) : ContainerStatus = copy(restartCount = f(restartCount))

  /** Returns a new data with ready set to new value */
  def withReady(value: Boolean) : ContainerStatus = copy(ready = value)
  /** transforms ready to result of function */
  def mapReady(f: Boolean => Boolean) : ContainerStatus = copy(ready = f(ready))

  /** Returns a new data with imageID set to new value */
  def withImageID(value: String) : ContainerStatus = copy(imageID = value)
  /** transforms imageID to result of function */
  def mapImageID(f: String => String) : ContainerStatus = copy(imageID = f(imageID))

  /** Returns a new data with state set to new value */
  def withState(value: io.k8s.api.core.v1.ContainerState) : ContainerStatus = copy(state = Some(value))
  /** if state has a value, transforms to the result of function*/
  def mapState(f: io.k8s.api.core.v1.ContainerState => io.k8s.api.core.v1.ContainerState) : ContainerStatus = copy(state = state.map(f))

  /** Returns a new data with allocatedResourcesStatus set to new value */
  def withAllocatedResourcesStatus(value: Seq[io.k8s.api.core.v1.ResourceStatus]) : ContainerStatus = copy(allocatedResourcesStatus = Some(value))
  /** Appends new values to allocatedResourcesStatus */
  def addAllocatedResourcesStatus(newValues: io.k8s.api.core.v1.ResourceStatus*) : ContainerStatus = copy(allocatedResourcesStatus = Some(allocatedResourcesStatus.fold(newValues)(_ ++ newValues)))
  /** if allocatedResourcesStatus has a value, transforms to the result of function*/
  def mapAllocatedResourcesStatus(f: Seq[io.k8s.api.core.v1.ResourceStatus] => Seq[io.k8s.api.core.v1.ResourceStatus]) : ContainerStatus = copy(allocatedResourcesStatus = allocatedResourcesStatus.map(f))

  /** Returns a new data with lastState set to new value */
  def withLastState(value: io.k8s.api.core.v1.ContainerState) : ContainerStatus = copy(lastState = Some(value))
  /** if lastState has a value, transforms to the result of function*/
  def mapLastState(f: io.k8s.api.core.v1.ContainerState => io.k8s.api.core.v1.ContainerState) : ContainerStatus = copy(lastState = lastState.map(f))

  /** Returns a new data with resources set to new value */
  def withResources(value: io.k8s.api.core.v1.ResourceRequirements) : ContainerStatus = copy(resources = Some(value))
  /** if resources has a value, transforms to the result of function*/
  def mapResources(f: io.k8s.api.core.v1.ResourceRequirements => io.k8s.api.core.v1.ResourceRequirements) : ContainerStatus = copy(resources = resources.map(f))

  /** Returns a new data with containerID set to new value */
  def withContainerID(value: String) : ContainerStatus = copy(containerID = Some(value))
  /** if containerID has a value, transforms to the result of function*/
  def mapContainerID(f: String => String) : ContainerStatus = copy(containerID = containerID.map(f))

  /** Returns a new data with started set to new value */
  def withStarted(value: Boolean) : ContainerStatus = copy(started = Some(value))
  /** if started has a value, transforms to the result of function*/
  def mapStarted(f: Boolean => Boolean) : ContainerStatus = copy(started = started.map(f))

  /** Returns a new data with user set to new value */
  def withUser(value: io.k8s.api.core.v1.ContainerUser) : ContainerStatus = copy(user = Some(value))
  /** if user has a value, transforms to the result of function*/
  def mapUser(f: io.k8s.api.core.v1.ContainerUser => io.k8s.api.core.v1.ContainerUser) : ContainerStatus = copy(user = user.map(f))

  /** Returns a new data with allocatedResources set to new value */
  def withAllocatedResources(value: Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]) : ContainerStatus = copy(allocatedResources = Some(value))
  /** Adds new values to allocatedResources */
  def addAllocatedResources(newValues: (String, io.k8s.apimachinery.pkg.api.resource.Quantity)*) : ContainerStatus = copy(allocatedResources = Some(allocatedResources.fold(newValues.toMap)(_ ++ newValues)))
  /** if allocatedResources has a value, transforms to the result of function*/
  def mapAllocatedResources(f: Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity] => Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]) : ContainerStatus = copy(allocatedResources = allocatedResources.map(f))

  /** Returns a new data with volumeMounts set to new value */
  def withVolumeMounts(value: Seq[io.k8s.api.core.v1.VolumeMountStatus]) : ContainerStatus = copy(volumeMounts = Some(value))
  /** Appends new values to volumeMounts */
  def addVolumeMounts(newValues: io.k8s.api.core.v1.VolumeMountStatus*) : ContainerStatus = copy(volumeMounts = Some(volumeMounts.fold(newValues)(_ ++ newValues)))
  /** if volumeMounts has a value, transforms to the result of function*/
  def mapVolumeMounts(f: Seq[io.k8s.api.core.v1.VolumeMountStatus] => Seq[io.k8s.api.core.v1.VolumeMountStatus]) : ContainerStatus = copy(volumeMounts = volumeMounts.map(f))
}

object ContainerStatus {

    implicit val encoder : Encoder[io.k8s.api.core.v1.ContainerStatus] = new Encoder[io.k8s.api.core.v1.ContainerStatus] {
        def apply[T : Builder](o: io.k8s.api.core.v1.ContainerStatus) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("name", o.name)
            .write("image", o.image)
            .write("restartCount", o.restartCount)
            .write("ready", o.ready)
            .write("imageID", o.imageID)
            .write("state", o.state)
            .write("allocatedResourcesStatus", o.allocatedResourcesStatus)
            .write("lastState", o.lastState)
            .write("resources", o.resources)
            .write("containerID", o.containerID)
            .write("started", o.started)
            .write("user", o.user)
            .write("allocatedResources", o.allocatedResources)
            .write("volumeMounts", o.volumeMounts)
            .build
        }
    }

    implicit val decoder: Decoder[ContainerStatus] = new Decoder[ContainerStatus] {
      def apply[T : Reader](t: T): Either[String, ContainerStatus] = for {
          obj <- ObjectReader(t)
          name <- obj.read[String]("name")
          image <- obj.read[String]("image")
          restartCount <- obj.read[Int]("restartCount")
          ready <- obj.read[Boolean]("ready")
          imageID <- obj.read[String]("imageID")
          state <- obj.readOpt[io.k8s.api.core.v1.ContainerState]("state")
          allocatedResourcesStatus <- obj.readOpt[Seq[io.k8s.api.core.v1.ResourceStatus]]("allocatedResourcesStatus")
          lastState <- obj.readOpt[io.k8s.api.core.v1.ContainerState]("lastState")
          resources <- obj.readOpt[io.k8s.api.core.v1.ResourceRequirements]("resources")
          containerID <- obj.readOpt[String]("containerID")
          started <- obj.readOpt[Boolean]("started")
          user <- obj.readOpt[io.k8s.api.core.v1.ContainerUser]("user")
          allocatedResources <- obj.readOpt[Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]]("allocatedResources")
          volumeMounts <- obj.readOpt[Seq[io.k8s.api.core.v1.VolumeMountStatus]]("volumeMounts")
      } yield ContainerStatus (
          name = name,
          image = image,
          restartCount = restartCount,
          ready = ready,
          imageID = imageID,
          state = state,
          allocatedResourcesStatus = allocatedResourcesStatus,
          lastState = lastState,
          resources = resources,
          containerID = containerID,
          started = started,
          user = user,
          allocatedResources = allocatedResources,
          volumeMounts = volumeMounts
        )
    }
}

