package io.k8s.api.authorization.v1

import dev.hnaderi.k8s.utils._

/** LabelSelectorAttributes indicates a label limited access. Webhook authors are encouraged to * ensure rawSelector and requirements are not both set * consider the requirements field if set * not try to parse or consider the rawSelector field if set. This is to avoid another CVE-2022-2880 (i.e. getting different systems to agree on how exactly to parse a query is not something we want), see https://www.oxeye.io/resources/golang-parameter-smuggling-attack for more details. For the *SubjectAccessReview endpoints of the kube-apiserver: * If rawSelector is empty and requirements are empty, the request is not limited. * If rawSelector is present and requirements are empty, the rawSelector will be parsed and limited if the parsing succeeds. * If rawSelector is empty and requirements are present, the requirements should be honored * If rawSelector is present and requirements are present, the request is invalid. */
final case class LabelSelectorAttributes(
  rawSelector : Option[String] = None,
  requirements : Option[Seq[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelectorRequirement]] = None
) {

  /** Returns a new data with rawSelector set to new value */
  def withRawSelector(value: String) : LabelSelectorAttributes = copy(rawSelector = Some(value))
  /** if rawSelector has a value, transforms to the result of function*/
  def mapRawSelector(f: String => String) : LabelSelectorAttributes = copy(rawSelector = rawSelector.map(f))

  /** Returns a new data with requirements set to new value */
  def withRequirements(value: Seq[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelectorRequirement]) : LabelSelectorAttributes = copy(requirements = Some(value))
  /** Appends new values to requirements */
  def addRequirements(newValues: io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelectorRequirement*) : LabelSelectorAttributes = copy(requirements = Some(requirements.fold(newValues)(_ ++ newValues)))
  /** if requirements has a value, transforms to the result of function*/
  def mapRequirements(f: Seq[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelectorRequirement] => Seq[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelectorRequirement]) : LabelSelectorAttributes = copy(requirements = requirements.map(f))
}

object LabelSelectorAttributes {

    implicit val encoder : Encoder[io.k8s.api.authorization.v1.LabelSelectorAttributes] = new Encoder[io.k8s.api.authorization.v1.LabelSelectorAttributes] {
        def apply[T : Builder](o: io.k8s.api.authorization.v1.LabelSelectorAttributes) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("rawSelector", o.rawSelector)
            .write("requirements", o.requirements)
            .build
        }
    }

    implicit val decoder: Decoder[LabelSelectorAttributes] = new Decoder[LabelSelectorAttributes] {
      def apply[T : Reader](t: T): Either[String, LabelSelectorAttributes] = for {
          obj <- ObjectReader(t)
          rawSelector <- obj.readOpt[String]("rawSelector")
          requirements <- obj.readOpt[Seq[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelectorRequirement]]("requirements")
      } yield LabelSelectorAttributes (
          rawSelector = rawSelector,
          requirements = requirements
        )
    }
}

