package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** Represents an NFS mount that lasts the lifetime of a pod. NFS volumes do not support ownership management or SELinux relabeling. */
final case class NFSVolumeSource(
  path : String,
  server : String,
  readOnly : Option[Boolean] = None
) {

  /** Returns a new data with path set to new value */
  def withPath(value: String) : NFSVolumeSource = copy(path = value)
  /** transforms path to result of function */
  def mapPath(f: String => String) : NFSVolumeSource = copy(path = f(path))

  /** Returns a new data with server set to new value */
  def withServer(value: String) : NFSVolumeSource = copy(server = value)
  /** transforms server to result of function */
  def mapServer(f: String => String) : NFSVolumeSource = copy(server = f(server))

  /** Returns a new data with readOnly set to new value */
  def withReadOnly(value: Boolean) : NFSVolumeSource = copy(readOnly = Some(value))
  /** if readOnly has a value, transforms to the result of function*/
  def mapReadOnly(f: Boolean => Boolean) : NFSVolumeSource = copy(readOnly = readOnly.map(f))
}

object NFSVolumeSource {

    implicit val encoder : Encoder[io.k8s.api.core.v1.NFSVolumeSource] = new Encoder[io.k8s.api.core.v1.NFSVolumeSource] {
        def apply[T : Builder](o: io.k8s.api.core.v1.NFSVolumeSource) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("path", o.path)
            .write("server", o.server)
            .write("readOnly", o.readOnly)
            .build
        }
    }

    implicit val decoder: Decoder[NFSVolumeSource] = new Decoder[NFSVolumeSource] {
      def apply[T : Reader](t: T): Either[String, NFSVolumeSource] = for {
          obj <- ObjectReader(t)
          path <- obj.read[String]("path")
          server <- obj.read[String]("server")
          readOnly <- obj.readOpt[Boolean]("readOnly")
      } yield NFSVolumeSource (
          path = path,
          server = server,
          readOnly = readOnly
        )
    }
}

