package io.k8s.api.authentication.v1

import dev.hnaderi.k8s.utils._

/** TokenRequestSpec contains client provided parameters of a token request. */
final case class TokenRequestSpec(
  audiences : Seq[String],
  boundObjectRef : Option[io.k8s.api.authentication.v1.BoundObjectReference] = None,
  expirationSeconds : Option[Int] = None
) {

  /** Returns a new data with audiences set to new value */
  def withAudiences(value: Seq[String]) : TokenRequestSpec = copy(audiences = value)
  /** Appends new values to audiences */
  def addAudiences(newValues: String*) : TokenRequestSpec = copy(audiences = audiences ++ newValues)
  /** transforms audiences to result of function */
  def mapAudiences(f: Seq[String] => Seq[String]) : TokenRequestSpec = copy(audiences = f(audiences))

  /** Returns a new data with boundObjectRef set to new value */
  def withBoundObjectRef(value: io.k8s.api.authentication.v1.BoundObjectReference) : TokenRequestSpec = copy(boundObjectRef = Some(value))
  /** if boundObjectRef has a value, transforms to the result of function*/
  def mapBoundObjectRef(f: io.k8s.api.authentication.v1.BoundObjectReference => io.k8s.api.authentication.v1.BoundObjectReference) : TokenRequestSpec = copy(boundObjectRef = boundObjectRef.map(f))

  /** Returns a new data with expirationSeconds set to new value */
  def withExpirationSeconds(value: Int) : TokenRequestSpec = copy(expirationSeconds = Some(value))
  /** if expirationSeconds has a value, transforms to the result of function*/
  def mapExpirationSeconds(f: Int => Int) : TokenRequestSpec = copy(expirationSeconds = expirationSeconds.map(f))
}

object TokenRequestSpec {

    implicit val encoder : Encoder[io.k8s.api.authentication.v1.TokenRequestSpec] = new Encoder[io.k8s.api.authentication.v1.TokenRequestSpec] {
        def apply[T : Builder](o: io.k8s.api.authentication.v1.TokenRequestSpec) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("audiences", o.audiences)
            .write("boundObjectRef", o.boundObjectRef)
            .write("expirationSeconds", o.expirationSeconds)
            .build
        }
    }

    implicit val decoder: Decoder[TokenRequestSpec] = new Decoder[TokenRequestSpec] {
      def apply[T : Reader](t: T): Either[String, TokenRequestSpec] = for {
          obj <- ObjectReader(t)
          audiences <- obj.read[Seq[String]]("audiences")
          boundObjectRef <- obj.readOpt[io.k8s.api.authentication.v1.BoundObjectReference]("boundObjectRef")
          expirationSeconds <- obj.readOpt[Int]("expirationSeconds")
      } yield TokenRequestSpec (
          audiences = audiences,
          boundObjectRef = boundObjectRef,
          expirationSeconds = expirationSeconds
        )
    }
}

