package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._


final case class TypedObjectReference(
  name : String,
  apiGroup : Option[String] = None,
  namespace : Option[String] = None
) {

  /** Returns a new data with name set to new value */
  def withName(value: String) : TypedObjectReference = copy(name = value)
  /** transforms name to result of function */
  def mapName(f: String => String) : TypedObjectReference = copy(name = f(name))

  /** Returns a new data with apiGroup set to new value */
  def withApiGroup(value: String) : TypedObjectReference = copy(apiGroup = Some(value))
  /** if apiGroup has a value, transforms to the result of function*/
  def mapApiGroup(f: String => String) : TypedObjectReference = copy(apiGroup = apiGroup.map(f))

  /** Returns a new data with namespace set to new value */
  def withNamespace(value: String) : TypedObjectReference = copy(namespace = Some(value))
  /** if namespace has a value, transforms to the result of function*/
  def mapNamespace(f: String => String) : TypedObjectReference = copy(namespace = namespace.map(f))
}

object TypedObjectReference {

    implicit val encoder : Encoder[io.k8s.api.core.v1.TypedObjectReference] = new Encoder[io.k8s.api.core.v1.TypedObjectReference] {
        def apply[T : Builder](o: io.k8s.api.core.v1.TypedObjectReference) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("name", o.name)
            .write("apiGroup", o.apiGroup)
            .write("namespace", o.namespace)
            .build
        }
    }

    implicit val decoder: Decoder[TypedObjectReference] = new Decoder[TypedObjectReference] {
      def apply[T : Reader](t: T): Either[String, TypedObjectReference] = for {
          obj <- ObjectReader(t)
          name <- obj.read[String]("name")
          apiGroup <- obj.readOpt[String]("apiGroup")
          namespace <- obj.readOpt[String]("namespace")
      } yield TypedObjectReference (
          name = name,
          apiGroup = apiGroup,
          namespace = namespace
        )
    }
}

